import {Moment} from 'moment';

import {TLD} from 'constants/tld';
import {URLs} from 'constants/urls';

import {WHEN_SPECIAL_VALUE, WhenType} from 'types/common/When';
import {ITrainsFilledSearchContext} from 'reducers/trains/context/types';
import {TTrainsSegmentId} from 'types/trains/common/segment/ITrainsSegment';
import {EDirection} from 'types/common/EDirection';

import {ROBOT} from 'utilities/dateUtils/formats';
import {internalUrl} from 'utilities/url';
import {
    DateLikeType,
    formatDate,
    isValidDate,
    parseDate,
} from 'utilities/dateUtils';
import {isWhenSpecialValue} from 'projects/trains/lib/context';
import {convertDirectionToPathDirection} from 'projects/trains/lib/urls/pathDirectionHelpers';
import {ITrainsSortingQuery} from 'projects/trains/lib/sort/getTrainsSortingQuery';
import {ITrainsFilterQuery} from 'projects/trains/lib/filters/getFiltersQuery';

export interface QueryInterface {
    [k: string]: string | number;
}

interface IOptions {
    withOrigin?: boolean;
    tld?: TLD;
}

/**
 * Возвращает дату, на которую нужно осуществить поиск, или специальное значение.
 */
function getPeriod(when: WhenType): string {
    if (isWhenSpecialValue(when) && when !== WHEN_SPECIAL_VALUE.ALL_DAYS) {
        return when;
    }

    if (when === WHEN_SPECIAL_VALUE.ALL_DAYS) {
        return '';
    }

    const period: Moment = parseDate(when);

    if (!isValidDate(period)) {
        return '';
    }

    return formatDate(period, ROBOT);
}

function parseReturnWhen(returnWhen?: DateLikeType | null): string | null {
    if (!returnWhen) {
        return null;
    }

    const parsedReturnWhen: Moment = parseDate(returnWhen);

    if (!isValidDate(parsedReturnWhen)) {
        return null;
    }

    return formatDate(parsedReturnWhen, ROBOT);
}

interface ITrainsContextSearchParams {
    from?: string;
    to: string;
    when?: WhenType;
    originalWhen?: WhenType;
    returnWhen?: DateLikeType | null;
}

interface ITrainsSearchParams {
    context: ITrainsContextSearchParams;
    filters?: ITrainsFilterQuery;
    sort?: ITrainsSortingQuery;
    lastSearchTimeMarker?: string;
    query?: QueryInterface;
    direction?: EDirection;
    forwardSegmentId?: TTrainsSegmentId | null;
}

export function prepareTrainsContextForSearchUrl(
    context: ITrainsFilledSearchContext,
): ITrainsContextSearchParams {
    return {
        ...context,
        from: context.from.slug,
        to: context.to.slug,
    };
}

/**
 * URL поиска поездов.
 * Если дата отправления не задана, то функция вернет ссылку на поиск на все дни.
 *
 * @param context - Упрощенный поисковый контекст.
 * @param context.from - Slug точки отправления.
 * @param context.to - Slug точки прибытия.
 * @param [context.when=''] - Дата отправления.
 * @param [filters] - Фильтры.
 * @param [sort] - Сортировка.
 * @param [direction] - Направление поезда.
 * @param [forwardSegmentId] - Выбранный сегмент туда.
 * @param [lastSearchTimeMarker] - Временная метка последнего поиска.
 * @param [query] - Дополнительные query-параметры.
 * @param [options] - Параметры формирования URL.
 * @param [options.withOrigin] - Добавлять origin в URL.
 * @param [options.tld] - TLD. Имеет смысл только если withOrigin в значении true.
 */
export function getTrainsSearchUrl(
    {
        context,
        filters,
        sort,
        lastSearchTimeMarker,
        query,
        direction,
        forwardSegmentId,
    }: ITrainsSearchParams,
    options?: IOptions,
): string {
    const {from, to, when = '', originalWhen = '', returnWhen} = context;
    const period: string = getPeriod(originalWhen || when);
    const parsedReturnWhen = parseReturnWhen(returnWhen);

    return internalUrl(
        `${URLs.trains}/${from}--${to}`,
        {
            when: period || null,
            returnWhen: parsedReturnWhen,
            direction: direction
                ? convertDirectionToPathDirection(direction)
                : undefined,
            forwardSegmentId:
                direction === EDirection.BACKWARD
                    ? forwardSegmentId
                    : undefined,
            lastSearchTimeMarker,
            ...query,
            ...filters,
            ...sort,
        },
        {
            ...options,
            filterNull: true,
        },
    );
}
