import {FC, memo, useCallback, useEffect, useMemo} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import {useParams} from 'react-router-dom';
import {ParsedQuery} from 'query-string';

import {EProjectName} from 'constants/common';

import {ELoadableReducer} from 'types/common/ELoadableReducer';
import {EFooterProject} from 'components/Footer/types';
import {ETrainsGoal} from 'utilities/metrika/types/goals/trains';

import directionReducer from 'reducers/trains/direction/reducer';
import requestDirectionSearchPageAction from 'reducers/trains/direction/thunk/requestDirectionSearchPageAction';

import isNotFoundSearchSelector from 'projects/trains/pages/DirectionSearchPage/selectors/isNotFoundSearchSelector';
import {directionSearchSelector} from 'selectors/trains/direction/searchSelector';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import useServerDataFetcher from 'utilities/hooks/useServerDataFetcher';
import {reachGoal} from 'utilities/metrika';
import useQuery from 'utilities/hooks/useQuery';
import {deviceMods} from 'utilities/stylesUtils';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import withReducers from 'containers/withReducers/withReducers';

import LayoutDefault from 'components/Layouts/LayoutDefault/LayoutDefault';
import BookLoader from 'components/BookLoader/BookLoader';
import CommonSearchError from 'projects/trains/components/CommonSearchError/CommonSearchError';
import Content from 'projects/trains/pages/DirectionSearchPage/components/Content/Content';
import NotFoundFragment from 'components/NotFound/NotFoundFragment/NotFoundFragment';

import requestDirectionSearchPageSSR from 'server/redux/trains/requestDirectionSearchPageSSR';

import {useDefaultDirectionPageSearchForm} from './hooks/useDefaultDirectionPageSearchForm';

import cx from './DirectionSearchPage.scss';

interface IParams {
    fromSlug: string;
    toSlug: string;
}

interface IDirectionSearchPageProps {}

const ROOT_QA = 'directionPage';

const DirectionSearchPage: FC<IDirectionSearchPageProps> = () => {
    const dispatch = useDispatch();
    const deviceType = useDeviceType();
    const {fromSlug, toSlug} = useParams<IParams>();
    const query = useQuery();

    const needToRender = useServerDataFetcher([requestDirectionSearchPageSSR]);

    const isNotFoundSearchContext = useSelector(isNotFoundSearchSelector);
    const directionSearchInfo = useSelector(directionSearchSelector);

    const handleFormSubmit = useCallback(() => {
        reachGoal(ETrainsGoal.DIRECTION_FORM_SUBMIT);
    }, []);

    const requestDirectionSearch = useCallback(
        (withRefreshIfErrors: boolean) => {
            dispatch(
                requestDirectionSearchPageAction({
                    params: {
                        fromSlug,
                        toSlug,
                    },
                    withRefreshIfErrors,
                }),
            );
        },
        [dispatch, fromSlug, toSlug],
    );

    const refreshSearchOnError = useCallback(() => {
        requestDirectionSearch(true);
    }, [requestDirectionSearch]);

    const content = useMemo(() => {
        if (isNotFoundSearchContext) {
            return <NotFoundFragment />;
        }

        if (directionSearchInfo.isFailed) {
            return <CommonSearchError onUpdate={refreshSearchOnError} />;
        }

        if (!directionSearchInfo.isFetched) {
            return <BookLoader isLoading />;
        }

        const {
            value: {searchDate, seoInfo, blocks},
        } = directionSearchInfo;

        return (
            <Content
                className={cx('content')}
                blocks={blocks}
                searchDate={searchDate}
                seoInfo={seoInfo}
                query={query as ParsedQuery}
                onFormSubmit={handleFormSubmit}
                {...prepareQaAttributes(ROOT_QA)}
            />
        );
    }, [
        directionSearchInfo,
        handleFormSubmit,
        isNotFoundSearchContext,
        query,
        refreshSearchOnError,
    ]);

    useDefaultDirectionPageSearchForm();

    useEffect(() => {
        requestDirectionSearch(false);
    }, [requestDirectionSearch]);

    useEffect(() => {
        reachGoal(ETrainsGoal.DIRECTION_PAGE_OPENED);
    }, []);

    if (!needToRender) {
        return null;
    }

    return (
        <LayoutDefault
            className={cx('root', deviceMods('root', deviceType))}
            footerClassName={cx('footer')}
            showSearchForm={deviceType.isDesktop}
            hasSideSheetNavigation={deviceType.isMobile}
            showNavigation
            searchFormIsStatic
            initiallyCalendarIsOpen={!isNotFoundSearchContext}
            searchFormInitialIsExpanded
            onSearchFormSubmit={handleFormSubmit}
            project={EProjectName.TRAINS}
            footerType={EFooterProject.TRAINS}
        >
            {content}
        </LayoutDefault>
    );
};

export default memo(
    withReducers([[ELoadableReducer.TRAINS_DIRECTION, directionReducer]])(
        DirectionSearchPage,
    ),
);
