import {FC, memo} from 'react';
import {ParsedQuery} from 'query-string';

import {EProjectName} from 'constants/common';

import {IWithClassName} from 'types/withClassName';
import {ISeoInfo} from 'types/hotels/common/ISeoInfo';
import {ITrainDirectionBlocks} from 'types/trains/seoPages/directionSearch/blocks';
import {ITrainDirectionSeoInfo} from 'types/trains/seoPages/directionSearch/seoInfo';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {
    IWithDataQaAttribute,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import {hotelsTitle} from 'i18n/common-crossLinks';

import Container from 'components/Layouts/Container/Container';
import SearchForm from 'projects/trains/components/SearchForm/SearchForm';
import CrossSaleHotelsBlock from 'components/CrossSaleHotelsBlock/CrossSaleHotelsBlock';
import FAQ from 'projects/trains/pages/DirectionSearchPage/components/Content/components/FAQ/FAQ';
import SearchPageFooter from 'projects/trains/components/search/SearchPageFooter/SearchPageFooter';
import Meta from 'projects/trains/pages/DirectionSearchPage/components/Content/components/Meta/Meta';
import Header from 'projects/trains/pages/DirectionSearchPage/components/Content/components/Header/Header';
import Reviews from 'projects/trains/pages/DirectionSearchPage/components/Content/components/Reviews/Reviews';
import Variants from 'projects/trains/pages/DirectionSearchPage/components/Content/components/Variants/Variants';
import CrossLinks from 'projects/trains/pages/DirectionSearchPage/components/Content/components/CrossLinks/CrossLinks';
import Description from 'projects/trains/pages/DirectionSearchPage/components/Content/components/Description/Description';

import cx from './Content.scss';

interface IContentProps extends IWithClassName, IWithDataQaAttribute {
    blocks: ITrainDirectionBlocks;
    searchDate: string | undefined;
    seoInfo: ISeoInfo<ITrainDirectionSeoInfo>;
    query: ParsedQuery;
    onFormSubmit(): void;
}

const Content: FC<IContentProps> = props => {
    const {
        className,
        blocks: {header, segments, info, faq, hotels},
        searchDate,
        seoInfo,
        query,
        onFormSubmit,
    } = props;

    const deviceType = useDeviceType();

    return (
        <Container
            className={cx('root', deviceMods('root', deviceType), className)}
        >
            <Meta seoInfo={seoInfo} />

            <Header
                className={cx('header')}
                title={header.title}
                text={header.text}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'header',
                })}
            />

            {deviceType.isMobile && (
                <SearchForm
                    className={cx('searchForm')}
                    query={query}
                    onSubmit={onFormSubmit}
                />
            )}

            {segments && searchDate && (
                <Variants
                    className={cx('variants')}
                    title={segments.title}
                    searchDate={searchDate}
                />
            )}

            {info && (
                <Description
                    className={cx('description')}
                    title={info.title}
                    text={info.text}
                    disclaimer={info.disclaimer}
                />
            )}

            {faq && <FAQ className={cx('faq')} items={faq.items} />}

            <CrossLinks className={cx('crossLinks')} />

            {hotels?.data && hotels.data.hasData && (
                <CrossSaleHotelsBlock
                    className={cx('crossLinks', 'crossLinks_hotels')}
                    vertical={EProjectName.TRAINS}
                    {...hotels.data}
                    title={hotelsTitle({
                        point: hotels.data.region.linguistics.prepositionalCase,
                    })}
                    linkType="region"
                />
            )}

            <Reviews className={cx('reviews')} />

            <SearchPageFooter className={cx('footer')} />
        </Container>
    );
};

export default memo(Content);
