import {memo, FC, useMemo} from 'react';
import {useSelector} from 'react-redux';

import {ISeoInfo} from 'types/hotels/common/ISeoInfo';
import {ITrainDirectionSeoInfo} from 'types/trains/seoPages/directionSearch/seoInfo';
import {isFilledTrainsSearchContext} from 'reducers/trains/context/types';

import {trainsContextSelector} from 'selectors/trains/trainsContextSelector';
import nonceSelector from 'selectors/common/nonceSelector';

import {getTrainsSearchCanonicalUrl} from 'projects/trains/lib/meta/searchMeta/getTrainsSearchCanonicalUrl';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import DocumentMeta from 'components/DocumentMeta/DocumentMeta';
import SocialSharingMeta from 'components/SocialSharingMeta/SocialSharingMeta';
import SchemaMarkup, {
    getAggregateOffer,
    getFAQSchemaMarkup,
} from 'components/SchemaMarkup/SchemaMarkup';

interface IMetaProps {
    seoInfo: ISeoInfo<ITrainDirectionSeoInfo>;
}

const Meta: FC<IMetaProps> = props => {
    const {
        seoInfo: {
            title,
            description,
            openGraph,
            schemaOrg: {offerData, faqItems},
        },
    } = props;

    const deviceType = useDeviceType();
    const context = useSelector(trainsContextSelector);
    const nonce = useSelector(nonceSelector);

    const canonicalUrl = useMemo(() => {
        if (!isFilledTrainsSearchContext(context)) {
            return null;
        }

        return getTrainsSearchCanonicalUrl({
            context,
            deviceType,
        });
    }, [context, deviceType]);

    const aggregateOffer = useMemo(() => {
        if (!canonicalUrl || !offerData) {
            return null;
        }

        return getAggregateOffer({
            ...offerData,
            url: canonicalUrl,
        });
    }, [canonicalUrl, offerData]);

    if (!canonicalUrl) {
        return null;
    }

    return (
        <>
            <DocumentMeta
                title={title}
                description={description}
                canonicalUrl={canonicalUrl}
            />

            <SocialSharingMeta
                url={canonicalUrl}
                title={openGraph.title}
                description={openGraph.description}
            />

            {aggregateOffer && (
                <SchemaMarkup data={aggregateOffer} nonce={nonce} />
            )}

            {Boolean(faqItems.length) && (
                <SchemaMarkup
                    data={getFAQSchemaMarkup(faqItems)}
                    nonce={nonce}
                />
            )}
        </>
    );
};

export default memo(Meta);
