import {FC, memo, useCallback, useEffect, useMemo} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import {times} from 'lodash';

import {IWithClassName} from 'types/withClassName';
import {isFilledTrainsSearchContext} from 'reducers/trains/context/types';
import {ESortType} from 'types/common/sort/ESortType';
import {EQueryingStatus} from 'types/trains/search/searchInfo/ITrainsSearchInfo';
import {ITrainsSearchSort} from 'projects/trains/lib/sort/types';
import {ETrainsGoal} from 'utilities/metrika/types/goals/trains';

import {
    stopTrainsSearch,
    trainsSearchActions,
} from 'reducers/trains/genericSearch/search/actions';

import {getTrainsOriginalSearchInfo} from 'selectors/trains/genericSearch/search/getTrainsOriginalSearchInfo';
import {trainsDirectionPageBadgesInfoSelector} from 'selectors/trains/genericSearch/badgesInfo/trainsBadgesInfoSelector';
import {trainsContextSelector} from 'selectors/trains/trainsContextSelector';
import {trainsOriginalVariantWithVisibleStatusByIdSelector} from 'selectors/trains/genericSearch/search/trainsSearchInfoSelector';
import {trainsSortVariants} from 'selectors/trains/genericSearch/sort/trainsSortVaraints';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useBoolean} from 'utilities/hooks/useBoolean';
import getHumanWhen from 'utilities/dateUtils/when/getHumanWhen';
import {deviceMods} from 'utilities/stylesUtils';
import {getNewNotFoundTitle} from 'projects/trains/lib/meta/searchMeta/getNotFoundTitle';
import {reachGoal} from 'utilities/metrika';

import * as i18n from 'i18n/trains-direction';

import TrainsSearchVariants from 'projects/trains/components/TrainsSearchVariants/TrainsSearchVariants';
import Heading from 'components/Heading/Heading';
import Button from 'components/Button/Button';
import Box from 'components/Box/Box';
import EmptySearch from 'projects/trains/components/EmptySearch/EmptySearch';
import SearchUpdateNotification from 'projects/trains/components/SearchUpdateNotiification/SearchUpdateNotification';
import TrainsSearchPageSegmentSkeleton from 'projects/trains/pages/TrainsSearchPage/components/TrainsSearchPagePreloader/TrainsSearchPageSegmentSkeleton/TrainsSearchPageSegmentSkeleton';

import cx from './Variants.scss';

interface IVariantsProps extends IWithClassName {
    title: string;
    searchDate: string;
}

const SORT: ITrainsSearchSort = {by: ESortType.DEPARTURE, isReverse: false};

const PREVIEW_SEGMENTS_COUNT = 6;
const BANNER_POSITION_INDEX = 4;

const Variants: FC<IVariantsProps> = props => {
    const {className, title, searchDate} = props;

    const {value: isExpanded, setTrue: showAllSegments} = useBoolean(false);

    const dispatch = useDispatch();
    const deviceType = useDeviceType();

    const directionContextSelector = useSelector(trainsContextSelector);
    const {variants, status} = useSelector(getTrainsOriginalSearchInfo);
    const badgesInfo = useSelector(trainsDirectionPageBadgesInfoSelector);

    const variantWithVisibleStatusById = useSelector(
        trainsOriginalVariantWithVisibleStatusByIdSelector,
    );

    const context = useMemo(() => {
        return {
            ...directionContextSelector,
            when: searchDate,
        };
    }, [directionContextSelector, searchDate]);

    const humanDate = useMemo(() => getHumanWhen(searchDate), [searchDate]);

    const sortedVariantIds = useMemo(() => {
        return trainsSortVariants({
            sortInfo: SORT,
            variants,
            context,
        }).map(variant => variant.id);
    }, [context, variants]);

    const handleShowUpdateNotification = useCallback(() => {
        reachGoal(ETrainsGoal.DIRECTION_EXPIRED);
    }, []);

    const handleOnUpdateNotification = useCallback(() => {
        reachGoal(ETrainsGoal.DIRECTION_EXPIRED_REFRESH_CLICK);
    }, []);

    const handleOnUpdateNotificationSkip = useCallback(() => {
        reachGoal(ETrainsGoal.DIRECTION_EXPIRED_CONTINUE);
    }, []);

    const items = useMemo(() => {
        if (!isFilledTrainsSearchContext(context)) {
            return null;
        }

        if (status === EQueryingStatus.QUERYING && !sortedVariantIds.length) {
            return (
                <Box className={cx('items')} between="3">
                    {times(PREVIEW_SEGMENTS_COUNT).map(index => (
                        <TrainsSearchPageSegmentSkeleton
                            key={index}
                            deviceType={deviceType}
                        />
                    ))}
                </Box>
            );
        }

        if (!humanDate) {
            return null;
        }

        return (
            <>
                <TrainsSearchVariants
                    className={cx('items')}
                    searchStatus={status}
                    badgesInfo={badgesInfo}
                    context={context}
                    sort={SORT}
                    variantIds={sortedVariantIds}
                    bannerIndexPosition={BANNER_POSITION_INDEX}
                    variantWithVisibleStatusById={variantWithVisibleStatusById}
                    visibleItemsCount={
                        isExpanded ? undefined : PREVIEW_SEGMENTS_COUNT
                    }
                />

                <SearchUpdateNotification
                    context={context}
                    date={humanDate}
                    onShowNotification={handleShowUpdateNotification}
                    onUpdate={handleOnUpdateNotification}
                    onSkip={handleOnUpdateNotificationSkip}
                />
            </>
        );
    }, [
        badgesInfo,
        context,
        deviceType,
        humanDate,
        isExpanded,
        sortedVariantIds,
        status,
        variantWithVisibleStatusById,
        handleOnUpdateNotification,
        handleOnUpdateNotificationSkip,
        handleShowUpdateNotification,
    ]);

    const content = useMemo(() => {
        if (!isFilledTrainsSearchContext(context)) {
            return null;
        }

        if (status === EQueryingStatus.DONE && !sortedVariantIds.length) {
            return (
                <EmptySearch
                    context={context}
                    {...getNewNotFoundTitle(context)}
                />
            );
        }

        return (
            <>
                <Heading className={cx('title')} level={2}>
                    {`${title} ${i18n.segmentsTitleDate({date: humanDate})}`}
                </Heading>

                {items}

                {!isExpanded &&
                    sortedVariantIds.length > PREVIEW_SEGMENTS_COUNT && (
                        <Button
                            className={cx('showAllSegmentsButton')}
                            size={deviceType.isMobile ? 'l' : 'm-inset'}
                            width={deviceType.isMobile ? 'max' : 'auto'}
                            onClick={showAllSegments}
                        >
                            {i18n.moreSegmentsButton({date: humanDate})}
                        </Button>
                    )}
            </>
        );
    }, [
        context,
        deviceType.isMobile,
        humanDate,
        isExpanded,
        items,
        showAllSegments,
        sortedVariantIds.length,
        status,
        title,
    ]);

    useEffect(() => {
        if (!isFilledTrainsSearchContext(context)) {
            return;
        }

        dispatch(trainsSearchActions.request(context));

        return (): void => {
            dispatch(stopTrainsSearch());
        };
    }, [context, dispatch]);

    return (
        <div className={cx('root', deviceMods('root', deviceType), className)}>
            {content}
        </div>
    );
};

export default memo(Variants);
