import {FC, memo, useEffect, useMemo} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import {useParams} from 'react-router-dom';

import {EProjectName} from 'constants/common';

import {ELoadableReducer} from 'types/common/ELoadableReducer';
import {
    EFirmTrainBlock,
    isFirmTrain,
} from 'types/trains/seoPages/firmTrain/TFirmTrainBlock';
import {EFooterProject} from 'components/Footer/types';

import firmTrainReducer from 'reducers/trains/firmTrain/reducer';
import requestFirmTrainThunkAction from 'reducers/trains/firmTrain/thunk/requestFirmTrainThunkAction';

import firmTrainSelector from 'selectors/trains/firmTrainSelector';

import useServerDataFetcher from 'utilities/hooks/useServerDataFetcher';
import prefetchData from 'projects/trains/pages/FirmTrainPage/utilities/prefetchData';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';

import * as i18nBlock from 'i18n/trainsFirmTrain';

import withReducers from 'containers/withReducers/withReducers';

import LayoutDefault, {
    ELayoutBgColor,
} from 'components/Layouts/LayoutDefault/LayoutDefault';
import BookLoader from 'components/BookLoader/BookLoader';
import SearchForm from 'projects/trains/pages/FirmTrainPage/components/SearchForm/SearchForm';
import Description from 'projects/trains/pages/FirmTrainPage/components/Description/Description';
import Advantages from 'projects/trains/pages/FirmTrainPage/components/Advantages/Advantages';
import Faq from 'projects/trains/pages/FirmTrainPage/components/Faq/Faq';
import LayoutError500 from 'components/Layouts/LayoutError500/LayoutError500';
import Helmet from 'projects/trains/pages/FirmTrainPage/components/Helmet/Helmet';
import {IErrorActionButton} from 'components/ErrorModal/ErrorModal';

import cx from './FirmTrainPage.scss';

const ERROR_ACTION: IErrorActionButton = {
    title: i18nBlock.fetchErrorButton(),
    type: 'button',
    handler: (): void => {
        document.location.reload();
    },
};

const FirmTrainPage: FC = () => {
    const needToRender = useServerDataFetcher([prefetchData]);
    const dispatch = useDispatch();
    const deviceType = useDeviceType();

    const {slug} = useParams<{slug: string}>();

    const firmTrainInfo = useSelector(firmTrainSelector);

    useEffect(() => {
        if (!isFirmTrain(slug)) {
            return;
        }

        dispatch(
            requestFirmTrainThunkAction({
                params: {
                    slug,
                },
            }),
        );
    }, [dispatch, slug]);

    const content = useMemo(() => {
        if (!isFirmTrain(slug)) {
            return;
        }

        if (firmTrainInfo.isFailed) {
            return (
                <LayoutError500
                    title={i18nBlock.fetchErrorTitle()}
                    subtitle={i18nBlock.fetchErrorSubtitle()}
                    action={ERROR_ACTION}
                />
            );
        }

        if (!firmTrainInfo.isFetched) {
            return <BookLoader isLoading />;
        }

        const {blocks, seoInfo} = firmTrainInfo.value;

        return (
            <>
                <Helmet slug={slug} seoInfo={seoInfo} />

                {blocks.map((block, index) => {
                    switch (block.type) {
                        case EFirmTrainBlock.SEARCH_FORM: {
                            return (
                                <SearchForm
                                    key={index}
                                    block={block}
                                    slug={slug}
                                />
                            );
                        }

                        case EFirmTrainBlock.DESCRIPTION: {
                            return <Description key={index} block={block} />;
                        }

                        case EFirmTrainBlock.ADVANTAGES: {
                            return <Advantages key={index} block={block} />;
                        }

                        case EFirmTrainBlock.FAQ: {
                            return <Faq key={index} block={block} />;
                        }

                        default: {
                            return null;
                        }
                    }
                })}
            </>
        );
    }, [
        firmTrainInfo.isFailed,
        firmTrainInfo.isFetched,
        firmTrainInfo.value,
        slug,
    ]);

    if (!needToRender) {
        return null;
    }

    return (
        <LayoutDefault
            className={cx('root', deviceMods('root', deviceType))}
            footerClassName={cx('footer')}
            project={EProjectName.TRAINS}
            footerType={EFooterProject.TRAINS}
            hasSideSheetNavigation
            showNavigation
            bgColor={ELayoutBgColor.LIGHT}
        >
            {content}
        </LayoutDefault>
    );
};

export default memo(
    withReducers([[ELoadableReducer.TRAINS_FIRM_TRAIN, firmTrainReducer]])(
        FirmTrainPage,
    ),
);
