import {memo, FC, useMemo} from 'react';
import {useSelector} from 'react-redux';

import firmTrainImagesMap from './constants/firmTrainImagesMap';

import {ISeoInfo} from 'types/hotels/common/ISeoInfo';
import {EFirmTrain} from 'types/trains/seoPages/firmTrain/TFirmTrainBlock';

import nonceSelector from 'selectors/common/nonceSelector';

import {
    getFAQSchemaMarkup,
    IFAQSchemaMarkupItem,
} from 'components/SchemaMarkup/utilities/getFAQSchemaMarkup';
import getFirmTrainPageUrl from 'projects/trains/lib/urls/getFirmTrainPageUrl';

import DocumentMeta from 'components/DocumentMeta/DocumentMeta';
import SocialSharingMeta from 'components/SocialSharingMeta/SocialSharingMeta';
import SchemaMarkup from 'components/SchemaMarkup/SchemaMarkup';

interface IHelmetProps {
    slug: EFirmTrain;
    seoInfo: ISeoInfo<{faqItems: IFAQSchemaMarkupItem[]}>;
}

const Helmet: FC<IHelmetProps> = props => {
    const {
        slug,
        seoInfo: {title, description, openGraph, schemaOrg},
    } = props;

    const nonce = useSelector(nonceSelector);

    const canonicalUrl = getFirmTrainPageUrl({
        slug,
        options: {withOrigin: true},
    });

    const alternates = useMemo(() => {
        return [
            {
                href: canonicalUrl,
                hreflang: 'ru',
            },
        ];
    }, [canonicalUrl]);

    return (
        <>
            <DocumentMeta
                title={title}
                description={description}
                canonicalUrl={canonicalUrl}
                alternates={alternates}
            />

            <SocialSharingMeta
                url={canonicalUrl}
                title={openGraph.title}
                description={openGraph.description}
                image={firmTrainImagesMap[slug]}
                imageHeight={openGraph.imageSize?.height ?? ''}
                imageWidth={openGraph.imageSize?.width ?? ''}
                twitterCardType="summary_large_image"
            />

            <SchemaMarkup
                data={getFAQSchemaMarkup(schemaOrg.faqItems)}
                nonce={nonce}
            />
        </>
    );
};

export default memo(Helmet);
