import React, {useMemo} from 'react';

import {EProjectName} from 'constants/common';

import {ITrainsHpGenericOrder} from 'reducers/trains/order/types';
import {ORDER_TYPE_VALUES} from 'projects/account/lib/orders/types';
import {IHappyPageCrossSale} from 'server/api/OrdersAPI/types/TOrdersGetOrderHappyPageApiResponse';

import {getOrderUrl} from 'projects/account/utilities/urls/getOrderUrl';
import getDownloadFilename from 'projects/account/utilities/downloadBlank/getDownloadFilename';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import getDownloadBlankUrl from 'projects/account/pages/Order/utilities/downloadBlank/getDownloadBlankUrl';

import * as i18nBlock from 'i18n/happyPage';

import HappyPageContent from 'src/projects/happyPage/components/HappyPageContent/HappyPageContent';
import OrderActions, {
    EOrderActionsSource,
} from 'components/OrderActions/OrderActions';

import GenericOrderInfo from '../GenericOrderInfo/GenericOrderInfo';

interface IGenericOrderHappyPageProps {
    order: ITrainsHpGenericOrder;
    crossSale: IHappyPageCrossSale;
    onOrderMount?(): void;
    onHotelsMount?(): void;
    onAllHotelsClick?(): void;
    onDetailsClick?(): void;
    onPromoCodeBlockMount?(): void;
    onPromoCodeBlockClick?(): void;
}

const ROOT_QA = 'happyPage';

const GenericOrderHappyPage: React.FC<IGenericOrderHappyPageProps> = ({
    order,
    crossSale,
    onOrderMount,
    onHotelsMount,
    onAllHotelsClick,
    onDetailsClick,
    onPromoCodeBlockMount,
    onPromoCodeBlockClick,
}) => {
    const prettyId = order.prettyId ?? '';

    const crossSaleProps = useMemo(() => {
        return {
            onHotelsMount,
            onAllHotelsClick,
            onPromoCodeBlockMount,
            onPromoCodeBlockClick,
        };
    }, [
        onAllHotelsClick,
        onHotelsMount,
        onPromoCodeBlockClick,
        onPromoCodeBlockMount,
    ]);

    const orderInfo = useMemo(() => {
        return (
            <GenericOrderInfo
                order={order}
                onMount={onOrderMount}
                {...prepareQaAttributes({
                    parent: ROOT_QA,
                    current: 'orderInfo',
                })}
            />
        );
    }, [order, onOrderMount]);

    const orderActions = useMemo(() => {
        const {id} = order;

        return (
            <OrderActions
                source={EOrderActionsSource.HAPPY_PAGE}
                orderDetails={{
                    url: getOrderUrl(id),
                    onClick: onDetailsClick,
                }}
                print={{
                    title: i18nBlock.print(),
                    url: getDownloadBlankUrl(
                        {id},
                        ORDER_TYPE_VALUES.TRAINS_GENERIC,
                    ),
                }}
                download={{
                    title: i18nBlock.download(),
                    url: getDownloadBlankUrl(
                        {id},
                        ORDER_TYPE_VALUES.TRAINS_GENERIC,
                    ),
                    fileName: getDownloadFilename('order', id, 'pdf'),
                }}
                {...prepareQaAttributes({
                    parent: ROOT_QA,
                    current: 'orderActions',
                })}
            />
        );
    }, [order, onDetailsClick]);

    return (
        <HappyPageContent
            orderType={EProjectName.TRAINS}
            orderInfo={orderInfo}
            orderActions={orderActions}
            crossSale={crossSale}
            prettyOrderId={prettyId}
            email={order.customerInfo.email}
            crossSaleProps={crossSaleProps}
            {...prepareQaAttributes(ROOT_QA)}
        />
    );
};

export default React.memo(GenericOrderHappyPage);
