import React, {useMemo} from 'react';

import {EProjectName} from 'constants/common';

import {ORDER_TYPE_VALUES} from 'projects/account/lib/orders/types';
import {ITrainsOrderDetails} from 'reducers/trains/order/types';
import {IHappyPageCrossSale} from 'server/api/OrdersAPI/types/TOrdersGetOrderHappyPageApiResponse';

import {getOrderUrl} from 'projects/account/utilities/urls/getOrderUrl';
import getDownloadFilename from 'projects/account/utilities/downloadBlank/getDownloadFilename';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import getDownloadBlankUrl from 'projects/account/pages/Order/utilities/downloadBlank/getDownloadBlankUrl';

import * as i18nBlock from 'i18n/happyPage';

import HappyPageContent from 'src/projects/happyPage/components/HappyPageContent/HappyPageContent';
import OrderInfo from 'projects/trains/pages/HappyPage/components/OrderInfo/OrderInfo';
import OrderActions, {
    EOrderActionsSource,
} from 'components/OrderActions/OrderActions';
import CardWithDeviceLayout from 'components/CardWithDeviceLayout/CardWithDeviceLayout';

interface ISingleOrderHappyPageProps {
    order: ITrainsOrderDetails;
    crossSale: IHappyPageCrossSale;
    onOrderMount?(): void;
    onHotelsMount?(): void;
    onAllHotelsClick?(): void;
    onDetailsClick?(): void;
    onPromoCodeBlockMount?(): void;
    onPromoCodeBlockClick?(): void;
}

const ROOT_QA = 'trainsHappyPage';

const SingleOrderHappyPage: React.FC<ISingleOrderHappyPageProps> = ({
    order,
    crossSale,
    onOrderMount,
    onHotelsMount,
    onAllHotelsClick,
    onDetailsClick,
    onPromoCodeBlockMount,
    onPromoCodeBlockClick,
}) => {
    const orderNumber = order.orderNumber ?? '';

    const crossSaleProps = useMemo(() => {
        return {
            onHotelsMount,
            onAllHotelsClick,
            onPromoCodeBlockMount,
            onPromoCodeBlockClick,
        };
    }, [
        onAllHotelsClick,
        onHotelsMount,
        onPromoCodeBlockClick,
        onPromoCodeBlockMount,
    ]);

    const orderInfo = useMemo(() => {
        return (
            <CardWithDeviceLayout>
                <OrderInfo order={order} onMount={onOrderMount} />
            </CardWithDeviceLayout>
        );
    }, [order, onOrderMount]);

    const orderActions = useMemo(() => {
        const {uid} = order;

        return (
            <OrderActions
                source={EOrderActionsSource.HAPPY_PAGE}
                orderDetails={{
                    url: getOrderUrl(uid),
                    onClick: onDetailsClick,
                }}
                print={{
                    title: i18nBlock.print(),
                    url: getDownloadBlankUrl(
                        {id: uid},
                        ORDER_TYPE_VALUES.TRAINS_GENERIC,
                    ),
                }}
                download={{
                    title: i18nBlock.download(),
                    url: getDownloadBlankUrl(
                        {id: uid},
                        ORDER_TYPE_VALUES.TRAINS_GENERIC,
                    ),
                    fileName: getDownloadFilename('order', uid, 'pdf'),
                }}
                {...prepareQaAttributes({
                    parent: ROOT_QA,
                    current: 'orderActions',
                })}
            />
        );
    }, [order, onDetailsClick]);

    return (
        <HappyPageContent
            orderType={EProjectName.TRAINS}
            orderInfo={orderInfo}
            orderActions={orderActions}
            crossSale={crossSale}
            prettyOrderId={orderNumber}
            email={order.userInfo.email}
            crossSaleProps={crossSaleProps}
            {...prepareQaAttributes(ROOT_QA)}
        />
    );
};

export default React.memo(SingleOrderHappyPage);
