import React, {useMemo} from 'react';
import {useSelector} from 'react-redux';

import {isFilledTrainsSearchContext} from 'reducers/trains/context/types';
import {ETrainsFilterType} from 'types/trains/search/filters/ITrainsFilters';

import {StoreInterface} from 'reducers/storeTypes';

import {
    trainsSortedVariantIdsSelector,
    trainsFilteredVariantsSelector,
    trainsFilteredVariantsInfoSelector,
} from 'selectors/trains/genericSearch/search/trainsSearchInfoSelector';
import {trainsContextSelector} from 'selectors/trains/trainsContextSelector';
import {trainsDisclaimersInfoSelector} from 'selectors/trains/genericSearch/disclaimersInfo/trainsDisclaimersInfoSelector';
import {trainsBadgesInfoSelector} from 'selectors/trains/genericSearch/badgesInfo/trainsBadgesInfoSelector';
import {getTrainsOriginalSearchInfo} from 'selectors/trains/genericSearch/search/getTrainsOriginalSearchInfo';
import {trainsSearchFiltersInfoSelector} from 'selectors/trains/genericSearch/filters/trainsSearchFiltersSelector';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceModMobile, deviceMods} from 'utilities/stylesUtils';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {checkTrainsSearchPageEmptySearch} from './utilities/checkTrainsSearchPageEmptySearch';
import {checkTrainsSearchPageEmptyResultAfterFiltration} from './utilities/checkTrainsSearchPageEmptyResultAfterFiltration';
import {getNewNotFoundTitle} from 'projects/trains/lib/meta/searchMeta/getNotFoundTitle';
import {useExperiments} from 'utilities/hooks/useExperiments';
import {checkHasNearestTrainDatesByDirection} from 'projects/trains/lib/genericSearch/nearestTrainDates/checkHasNearestTrainDatesByDirection';

import Box from 'components/Box/Box';
import Flex from 'components/Flex/Flex';
import TrainsSearchPagePreloader from 'projects/trains/pages/TrainsSearchPage/components/TrainsSearchPagePreloader/TrainsSearchPagePreloader';
import EmptySearch from 'projects/trains/components/EmptySearch/EmptySearch';
import TrainsSearchVariants from 'projects/trains/components/TrainsSearchVariants/TrainsSearchVariants';
import TrainsSearchPageSort from 'projects/trains/pages/TrainsSearchPage/components/TrainsSearchPageSort/TrainsSearchPageSort';
import TrainsHideWithoutPriceFilter from '../TrainsSearchPageFilters/components/TrainsHideWithoutPriceFilter/TrainsHideWithoutPriceFilter';

import {useTrainsSearchPageLoaderState} from 'projects/trains/pages/TrainsSearchPage/hooks/useTrainsSearchPageLoaderState';

import TrainsSearchPagePinnedSegment from '../TrainsSearchPagePinnedSegment/TrainsSearchPagePinnedSegment';
import TrainsSearchPageNearestTrainDates from '../TrainsSearchPageNearestTrainDates/TrainsSearchPageNearestTrainDates';
import TrainsSearchPageHeaderDisclaimers from '../TrainsSearchPageHeaderDisclaimers/TrainsSearchPageHeaderDisclaimers';
import TrainsSearchPageUpdateNotification from '../TrainsSearchPageUpdateNotification/TrainsSearchPageUpdateNotification';
import TrainsSearchPageTitle from '../TrainsSearchPageTitle/TrainsSearchPageTitle';
import TrainsSearchPageEmptyResultAfterFiltration from '../TrainsSearchPageEmptyResultAfterFiltration/TrainsSearchPageEmptyResultAfterFiltration';

import cx from './TrainsSearchPageContent.scss';

const VARIANTS_ADFOX_INDEX_POSITION = 4;

interface ITrainsSearchPageContentProps {}

// Добавить общий TrainsSearchPageTitle для desktop and mobile
const TrainsSearchPageContent: React.FC<ITrainsSearchPageContentProps> = () => {
    const deviceType = useDeviceType();
    const {
        status,
        errorCode,
        variants: originalVariants,
        nearestTrainDatesByDirection,
    } = useSelector(getTrainsOriginalSearchInfo);
    const sortedVariantIds = useSelector(trainsSortedVariantIdsSelector);
    const filteredVariants = useSelector(trainsFilteredVariantsSelector);
    const {variantWithVisibleStatusById} = useSelector(
        trainsFilteredVariantsInfoSelector,
    );
    const sort = useSelector(
        (state: StoreInterface) => state.trains.search.sort,
    );
    const filters = useSelector(trainsSearchFiltersInfoSelector);
    const searchContext = useSelector(trainsContextSelector);
    const disclaimersInfo = useSelector(trainsDisclaimersInfoSelector);
    const badgesInfo = useSelector(trainsBadgesInfoSelector);
    const hasPageLoader = useTrainsSearchPageLoaderState();
    const hasEmptySearch = checkTrainsSearchPageEmptySearch({
        status,
        errorCode,
        originalVariants,
    });
    const hasEmptyResultAfterFiltration =
        checkTrainsSearchPageEmptyResultAfterFiltration({
            filteredVariants,
            originalVariants,
        });
    const {trainsNearestTrainDates: trainsNearestTrainDatesEnabled} =
        useExperiments();

    const nearestTrainDatesNode = useMemo(() => {
        if (
            !isFilledTrainsSearchContext(searchContext) ||
            !trainsNearestTrainDatesEnabled
        ) {
            return null;
        }

        const hasNearestTrainDatesByDirection =
            checkHasNearestTrainDatesByDirection({
                direction: searchContext.direction,
                nearestTrainDatesByDirection,
                isRoundTrip: Boolean(searchContext.returnWhen),
            });

        if (!hasNearestTrainDatesByDirection) {
            return null;
        }

        return (
            <TrainsSearchPageNearestTrainDates
                context={searchContext}
                direction={searchContext.direction}
                aboveContainer={deviceType.isMobile ? 2 : 4}
                belowContainer={deviceType.isMobile ? 7 : 15}
                nearestTrainDatesByDirection={nearestTrainDatesByDirection}
            />
        );
    }, [
        searchContext,
        deviceType,
        nearestTrainDatesByDirection,
        trainsNearestTrainDatesEnabled,
    ]);

    if (hasPageLoader || !isFilledTrainsSearchContext(searchContext)) {
        return (
            <TrainsSearchPagePreloader
                deviceType={deviceType}
                context={searchContext}
                {...prepareQaAttributes('searchPreloader')}
            />
        );
    }

    if (hasEmptyResultAfterFiltration) {
        return <TrainsSearchPageEmptyResultAfterFiltration />;
    }

    if (hasEmptySearch) {
        if (nearestTrainDatesNode) {
            return nearestTrainDatesNode;
        }

        return (
            <EmptySearch
                className={cx(deviceMods('emptySearch', deviceType))}
                context={searchContext}
                sort={sort}
                {...getNewNotFoundTitle(searchContext)}
            />
        );
    }

    return (
        <>
            <TrainsSearchPageHeaderDisclaimers
                className={cx('disclaimer')}
                context={searchContext}
                disclaimersInfo={disclaimersInfo}
            />
            {nearestTrainDatesNode}
            <TrainsSearchPagePinnedSegment
                searchStatus={status}
                context={searchContext}
                variants={originalVariants}
            />
            {deviceType.isMobile &&
                (searchContext.returnWhen || nearestTrainDatesNode) && (
                    <TrainsSearchPageTitle
                        className={cx(deviceModMobile('title', deviceType))}
                        variants={originalVariants}
                        context={searchContext}
                        nearestTrainDatesByDirection={
                            nearestTrainDatesByDirection
                        }
                        {...prepareQaAttributes('trainsSearchHeader')}
                    />
                )}
            {deviceType.isDesktop && (
                <Box above={5} between={5}>
                    <TrainsSearchPageTitle
                        variants={originalVariants}
                        context={searchContext}
                        nearestTrainDatesByDirection={
                            nearestTrainDatesByDirection
                        }
                        {...prepareQaAttributes('trainsSearchHeader')}
                    />
                    <Flex
                        inline
                        justifyContent="space-between"
                        className={cx('searchSortToolbar')}
                        {...prepareQaAttributes('searchToolbar')}
                    >
                        {sortedVariantIds.length > 1 && (
                            <TrainsSearchPageSort
                                sort={sort}
                                isFetched
                                {...prepareQaAttributes({
                                    parent: 'searchToolbar',
                                    current: 'sorting',
                                })}
                            />
                        )}
                        {filters && (
                            <TrainsHideWithoutPriceFilter
                                className={cx('hideWithoutPriceFilter')}
                                filter={
                                    filters[
                                        ETrainsFilterType.HIDE_WITHOUT_PRICE
                                    ]
                                }
                            />
                        )}
                    </Flex>
                </Box>
            )}
            <TrainsSearchVariants
                className={cx('variants', deviceMods('variants', deviceType))}
                searchStatus={status}
                badgesInfo={badgesInfo}
                context={searchContext}
                sort={sort}
                variantIds={sortedVariantIds}
                bannerIndexPosition={VARIANTS_ADFOX_INDEX_POSITION}
                variantWithVisibleStatusById={variantWithVisibleStatusById}
            />
            <TrainsSearchPageUpdateNotification context={searchContext} />
        </>
    );
};

export default TrainsSearchPageContent;
