import React, {useCallback} from 'react';
import {useSelector, useDispatch} from 'react-redux';

import {mapSearchFastFilterToClickGoal} from 'projects/trains/constants/metrika';

import {isFilledTrainsSearchContext} from 'reducers/trains/context/types';
import {EQueryingStatus} from 'types/trains/search/searchInfo/ITrainsSearchInfo';
import {TTrainsFilterTypeAndValue} from 'types/trains/search/filters/TTrainsFilterTypeAndValue';

import {setTrainsSearchFilterValue} from 'reducers/trains/genericSearch/filters/actions';

import {trainsContextSelector} from 'selectors/trains/trainsContextSelector';
import {trainsSearchFiltersInfoSelector} from 'selectors/trains/genericSearch/filters/trainsSearchFiltersSelector';
import {getTrainsOriginalSearchInfo} from 'selectors/trains/genericSearch/search/getTrainsOriginalSearchInfo';

import {reachGoal} from 'utilities/metrika';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {IWithQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {checkTrainsSearchPageEmptySearch} from 'projects/trains/pages/TrainsSearchPage/components/TrainsSearchPageContent/utilities/checkTrainsSearchPageEmptySearch';

import {useTrainsSearchPageLoaderState} from 'projects/trains/pages/TrainsSearchPage/hooks/useTrainsSearchPageLoaderState';

import TrainsSearchPageMobileFastFilters from './TrainsSearchPageMobileFastFilters/TrainsSearchPageMobileFastFilters';

interface ITrainsSearchPageFastFiltersProps extends IWithQaAttributes {}

const TrainsSearchPageFastFilters: React.FC<ITrainsSearchPageFastFiltersProps> =
    () => {
        const dispatch = useDispatch();
        const deviceType = useDeviceType();
        const hasPageLoader = useTrainsSearchPageLoaderState();
        const searchContext = useSelector(trainsContextSelector);
        const filters = useSelector(trainsSearchFiltersInfoSelector);
        const {
            status,
            errorCode,
            variants: originalVariants,
        } = useSelector(getTrainsOriginalSearchInfo);
        const hasEmptySearch = checkTrainsSearchPageEmptySearch({
            status,
            errorCode,
            originalVariants,
        });

        const handleChangeFilterValue = useCallback(
            (typeAndValue: TTrainsFilterTypeAndValue) => {
                reachGoal(mapSearchFastFilterToClickGoal[typeAndValue.type]);

                dispatch(setTrainsSearchFilterValue(typeAndValue));
            },
            [dispatch],
        );

        if (hasEmptySearch || !isFilledTrainsSearchContext(searchContext)) {
            return null;
        }

        if (deviceType.isMobile) {
            return (
                <TrainsSearchPageMobileFastFilters
                    filters={filters}
                    hasPageLoader={hasPageLoader}
                    onChangeFilterValue={handleChangeFilterValue}
                    searchIsFinished={status === EQueryingStatus.DONE}
                />
            );
        }

        return null;
    };

export default TrainsSearchPageFastFilters;
