import {FC, ReactNode, useCallback, useEffect, useMemo} from 'react';

import {
    ETrainsFilterType,
    ITrainsSimpleFilterOption,
    ITrainsHighSpeedTrainFilter,
} from 'types/trains/search/filters/ITrainsFilters';
import {IWithClassName} from 'types/withClassName';
import {TTrainsFilterTypeAndValue} from 'types/trains/search/filters/TTrainsFilterTypeAndValue';
import {ETrainsGoal} from 'utilities/metrika/types/goals/trains';

import {reachGoal} from 'utilities/metrika';
import {clearHighSpeedTrainName} from './utilities/clearHighSpeedTrainName';
import {getHighSpeedTrainFastFilterOptions} from './utilities/getHighSpeedTrainFastFilterOptions';
import {checkCanRenderHighSpeedTrainFastFilter} from './utilities/checkCanRenderHighSpeedTrainFastFilter';
import toggleValue from 'projects/trains/pages/TrainsSearchPage/components/TrainsSearchPageFilters/utilities/toggleTrainsFilterValue';
import {convertFilterOptionToValue} from 'projects/trains/pages/TrainsSearchPage/components/TrainsSearchPageFilters/utilities/convertTrainSimpleFilterOption';

import * as i18nBlock from 'i18n/trains-filters';

import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import Price from 'components/Price/Price';
import CheckButton from 'components/CheckButton/CheckButton';
import TextSkeleton from 'components/Skeletons/TextSkeleton/TextSkeleton';

import cx from './TrainsHighSpeedTrainFastFilter.scss';

interface ITrainsHighSpeedTrainFastFilterProps extends IWithClassName {
    searchIsFinished: boolean;
    filter: ITrainsHighSpeedTrainFilter;
    onChange(typeAndValue: TTrainsFilterTypeAndValue): void;
}

const TrainsHighSpeedTrainFastFilter: FC<ITrainsHighSpeedTrainFastFilterProps> =
    props => {
        const {className, filter, onChange, searchIsFinished} = props;
        const {
            value: filterValue,
            activeOptions: filterActiveOptions,
            optionMinPriceList: filterOptionMinPriceList,
        } = filter;
        const filterOptions = getHighSpeedTrainFastFilterOptions(filter);
        const handleChange = useCallback(
            (option: ITrainsSimpleFilterOption) => {
                const toggledValue = toggleValue(filterValue, option.value);

                onChange({
                    type: ETrainsFilterType.HIGH_SPEED_TRAIN,
                    value: toggledValue,
                });
            },
            [filterValue, onChange],
        );

        const filterAvailableValues = useMemo(() => {
            return filterActiveOptions.map(option => option.value);
        }, [filterActiveOptions]);

        const renderOptionAside = useCallback(
            (
                option: ITrainsSimpleFilterOption,
                index,
                isDisabled,
            ): ReactNode => {
                const optionMinPrice = filterOptionMinPriceList?.[index];
                const price = optionMinPrice?.price;

                if (isDisabled) {
                    return (
                        <Text size="xs" color="disabled">
                            {i18nBlock.emptyDashFastDashFilters()}
                        </Text>
                    );
                }

                if (!price) {
                    if (searchIsFinished) {
                        return (
                            <Text size="xs" color="secondary">
                                {i18nBlock.withoutDashPriceDashFastDashFilters()}
                            </Text>
                        );
                    }

                    return (
                        <TextSkeleton
                            className={cx('priceSkeleton')}
                            withAnimation
                            size="xxs"
                        />
                    );
                }

                return (
                    <Text size="xs" color="secondary">
                        <Price isFrom isRound {...price} />
                    </Text>
                );
            },
            [filterOptionMinPriceList, searchIsFinished],
        );

        const canRenderFilter = useMemo(
            () => checkCanRenderHighSpeedTrainFastFilter(filter),
            [filter],
        );

        useEffect(() => {
            if (canRenderFilter) {
                reachGoal(ETrainsGoal.FAST_FILTER_HIGH_SPEED_TRAIN_SHOW);
            }
        }, [canRenderFilter]);

        if (!canRenderFilter) {
            return null;
        }

        return (
            <Flex
                className={className}
                flexDirection="row"
                inline
                between={2}
                nowrap
            >
                {filterOptions.map((option, index) => {
                    const value = convertFilterOptionToValue(option);
                    const valueStr = String(value);
                    const isChecked = filterValue.includes(value);
                    const isDisabled = filterAvailableValues
                        ? !isChecked && !filterAvailableValues.includes(value)
                        : false;

                    return (
                        <CheckButton
                            key={valueStr}
                            size="l"
                            width="max"
                            checked={isChecked}
                            disabled={isDisabled}
                            onClick={(): void => handleChange(option)}
                        >
                            <Flex
                                className={cx('checkButton')}
                                flexDirection="column"
                                alignItems="center"
                            >
                                <Text
                                    size="s"
                                    color={isDisabled ? 'disabled' : 'primary'}
                                >
                                    {clearHighSpeedTrainName(option)}
                                </Text>
                                {renderOptionAside(option, index, isDisabled)}
                            </Flex>
                        </CheckButton>
                    );
                })}
            </Flex>
        );
    };

export default TrainsHighSpeedTrainFastFilter;
