import React from 'react';

import {
    ITrainsFilters,
    ETrainsFilterType,
} from 'types/trains/search/filters/ITrainsFilters';
import {TTrainsFilterTypeAndValue} from 'types/trains/search/filters/TTrainsFilterTypeAndValue';

import getFilterIsActive from 'projects/trains/lib/filters/getFilterIsActive';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nCommonFiltersBlock from 'i18n/common-filters';
import * as i18nTrainsFiltersBlock from 'i18n/trains-filters';

import CommonFiltersDesktop from 'components/FiltersDesktop/FiltersDesktop';
import DropdownFilter from 'components/DropdownFilter/DropdownFilter';
import TrainsPriceFilter from '../components/TrainsPriceFilter/TrainsPriceFilter';
import TrainsTariffClassFilter from '../components/TrainsTariffClassFilter/TrainsTariffClassFilter';
import TrainsHighSpeedTrainFilter from '../components/TrainsHighSpeedTrainFilter/TrainsHighSpeedTrainFilter';
import TrainsDepartureAndArrivalTimeFilter from '../components/TrainsDepartureAndArrivalTimeFilter/TrainsDepartureAndArrivalTimeFilter';
import TrainsStationFromAndStationToFilter from '../components/TrainsStationFromAndStationToFilter/TrainsStationFromAndStationToFilter';

import cx from './TrainsSearchPageDesktopFilters.scss';

interface ITrainsSearchPageDesktopFiltersProps extends IWithQaAttributes {
    filters: ITrainsFilters | null;
    isLoading: boolean;
    canReset: boolean;
    onReset(): void;
    onChangeFilterValue(typeAndValue: TTrainsFilterTypeAndValue): void;
}

const SKELETONS_WIDTH = [83, 132, 169, 242, 102];

const TrainsSearchPageDesktopFilters: React.FC<ITrainsSearchPageDesktopFiltersProps> =
    props => {
        const {filters, canReset, onReset, isLoading, onChangeFilterValue} =
            props;

        let filtersNode: React.ReactNode = null;

        if (filters) {
            const {
                [ETrainsFilterType.ARRIVAL]: arrivalFilter,
                [ETrainsFilterType.DEPARTURE]: departureFilter,
                [ETrainsFilterType.HIGH_SPEED_TRAIN]: highSpeedTrainFilter,
                [ETrainsFilterType.PRICE_RANGE]: priceRangeFilter,
                [ETrainsFilterType.STATION_FROM]: stationFromFilter,
                [ETrainsFilterType.STATION_TO]: stationToFilter,
                [ETrainsFilterType.TRAIN_TARIFF_CLASS]: trainTariffClassFilter,
            } = filters;

            filtersNode = (
                <>
                    {priceRangeFilter.availableWithOptions && (
                        <DropdownFilter
                            popupClassName={cx('price')}
                            label={i18nCommonFiltersBlock.price()}
                            checked={getFilterIsActive(priceRangeFilter)}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'priceTrigger',
                            })}
                        >
                            <TrainsPriceFilter
                                filter={priceRangeFilter}
                                onChange={onChangeFilterValue}
                            />
                        </DropdownFilter>
                    )}
                    {trainTariffClassFilter.availableWithOptions && (
                        <DropdownFilter
                            label={i18nTrainsFiltersBlock.titleDashTrainTariffClassDashRedesign()}
                            checked={getFilterIsActive(trainTariffClassFilter)}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'trainTariffClassTrigger',
                            })}
                        >
                            <TrainsTariffClassFilter
                                className={cx('tariffClass')}
                                filter={trainTariffClassFilter}
                                onChange={onChangeFilterValue}
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: 'trainTariffClass',
                                })}
                            />
                        </DropdownFilter>
                    )}
                    {highSpeedTrainFilter.availableWithOptions && (
                        <DropdownFilter
                            label={i18nTrainsFiltersBlock.titleDashHighSpeedTrain()}
                            checked={getFilterIsActive(highSpeedTrainFilter)}
                        >
                            <TrainsHighSpeedTrainFilter
                                className={cx('highSpeedTrain')}
                                filter={highSpeedTrainFilter}
                                onChange={onChangeFilterValue}
                            />
                        </DropdownFilter>
                    )}
                    {Boolean(
                        departureFilter.availableWithOptions ||
                            arrivalFilter.availableWithOptions,
                    ) && (
                        <DropdownFilter
                            label={i18nCommonFiltersBlock.departureAndArrivalTime()}
                            checked={
                                getFilterIsActive(departureFilter) ||
                                getFilterIsActive(arrivalFilter)
                            }
                        >
                            <TrainsDepartureAndArrivalTimeFilter
                                arrivalFilter={arrivalFilter}
                                departureFilter={departureFilter}
                                onChange={onChangeFilterValue}
                            />
                        </DropdownFilter>
                    )}
                    {Boolean(
                        stationFromFilter.availableWithOptions ||
                            stationToFilter.availableWithOptions,
                    ) && (
                        <DropdownFilter
                            label={i18nTrainsFiltersBlock.buttonDashDefaultDashTitleDashStationFrom()}
                            checked={
                                getFilterIsActive(stationFromFilter) ||
                                getFilterIsActive(stationToFilter)
                            }
                        >
                            <TrainsStationFromAndStationToFilter
                                stationFromFilter={stationFromFilter}
                                stationToFilter={stationToFilter}
                                onChange={onChangeFilterValue}
                            />
                        </DropdownFilter>
                    )}
                </>
            );
        }

        return (
            <CommonFiltersDesktop
                skeletonsWidth={SKELETONS_WIDTH}
                loading={isLoading}
                canReset={canReset}
                onReset={onReset}
                withShadow={false}
                withoutHorizontalPaddings
                {...prepareQaAttributes(props)}
            >
                {filtersNode}
            </CommonFiltersDesktop>
        );
    };

export default TrainsSearchPageDesktopFilters;
