import React, {useCallback} from 'react';
import {useSelector, useDispatch} from 'react-redux';

import {mapSearchFilterToClickGoal} from 'projects/trains/constants/metrika';

import {TTrainsFilterTypeAndValue} from 'types/trains/search/filters/TTrainsFilterTypeAndValue';
import {ITrainsSearchSort} from 'projects/trains/lib/sort/types';
import {isFilledTrainsSearchContext} from 'reducers/trains/context/types';
import {ETrainsGoal} from 'utilities/metrika/types/goals/trains';

import {
    setSorting,
    changeSortReachGoal,
} from 'reducers/trains/search/actions/sorting';
import {
    setTrainsSearchFilterValue,
    resetAllTrainsSearchFilterValues,
} from 'reducers/trains/genericSearch/filters/actions';

import {trainsContextSelector} from 'selectors/trains/trainsContextSelector';
import {getTrainsSearchSortInfo} from 'selectors/trains/genericSearch/sort/trainsSearchSortInfoSelector';
import {getTrainsOriginalSearchInfo} from 'selectors/trains/genericSearch/search/getTrainsOriginalSearchInfo';
import {trainsFilteredVariantsSelector} from 'selectors/trains/genericSearch/search/trainsSearchInfoSelector';
import {trainsSearchFiltersInfoSelector} from 'selectors/trains/genericSearch/filters/trainsSearchFiltersSelector';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {anyFiltersAvailable} from 'projects/trains/lib/filters/anyFiltersAvailable';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {reachGoal} from 'utilities/metrika';
import {checkTrainsSearchPageEmptySearch} from 'projects/trains/pages/TrainsSearchPage/components/TrainsSearchPageContent/utilities/checkTrainsSearchPageEmptySearch';
import {getActiveFilterManagerTypes} from 'projects/trains/lib/genericSearch/filters/managers/getActiveFilterManagerTypes';

import {useTrainsSearchPageLoaderState} from 'projects/trains/pages/TrainsSearchPage/hooks/useTrainsSearchPageLoaderState';

import TrainsSearchPageDesktopFilters from './TrainsSearchPageDesktopFilters/TrainsSearchPageDesktopFilters';
import TrainsSearchPageMobileFilters from './TrainsSearchPageMobileFilters/TrainsSearchPageMobileFilters';

interface ITrainsSearchPageFiltersProps {}

const TrainsSearchPageFilters: React.FC<ITrainsSearchPageFiltersProps> = () => {
    const dispatch = useDispatch();
    const deviceType = useDeviceType();
    const isLoading = useTrainsSearchPageLoaderState();
    const sort = useSelector(getTrainsSearchSortInfo);
    const searchContext = useSelector(trainsContextSelector);
    const filters = useSelector(trainsSearchFiltersInfoSelector);
    const filteredVariants = useSelector(trainsFilteredVariantsSelector);
    const {
        status,
        errorCode,
        variants: originalVariants,
    } = useSelector(getTrainsOriginalSearchInfo);
    const hasEmptySearch = checkTrainsSearchPageEmptySearch({
        status,
        errorCode,
        originalVariants,
    });

    const handleChangeFilterValue = useCallback(
        (typeAndValue: TTrainsFilterTypeAndValue) => {
            reachGoal(mapSearchFilterToClickGoal[typeAndValue.type]);

            dispatch(setTrainsSearchFilterValue(typeAndValue));
        },
        [dispatch],
    );

    const handleResetFilters = useCallback(() => {
        reachGoal(ETrainsGoal.FILTER_RESET_CLICK);

        dispatch(resetAllTrainsSearchFilterValues());
    }, [dispatch]);

    const handleChangeSorting = useCallback(
        (sortToSet: ITrainsSearchSort) => {
            changeSortReachGoal({sort, sortToSet});

            dispatch(setSorting(sortToSet));
        },
        [dispatch, sort],
    );

    const isResetButtonAvailable = Boolean(
        filters && getActiveFilterManagerTypes({filters}).length,
    );

    if (hasEmptySearch || !isFilledTrainsSearchContext(searchContext)) {
        return null;
    }

    if (filters && !anyFiltersAvailable(filters)) {
        return null;
    }

    if (deviceType.isMobile) {
        return (
            <TrainsSearchPageMobileFilters
                filters={filters}
                variantsCount={filteredVariants.length}
                sort={sort}
                isLoading={isLoading}
                canReset={isResetButtonAvailable}
                onReset={handleResetFilters}
                onChangeFilterValue={handleChangeFilterValue}
                onChangeSorting={handleChangeSorting}
                {...prepareQaAttributes('mobileFilters')}
            />
        );
    }

    return (
        <TrainsSearchPageDesktopFilters
            isLoading={isLoading}
            canReset={isResetButtonAvailable}
            filters={filters}
            onReset={handleResetFilters}
            onChangeFilterValue={handleChangeFilterValue}
            {...prepareQaAttributes('inlineFilters')}
        />
    );
};

export default TrainsSearchPageFilters;
