import React, {useCallback, useMemo} from 'react';

import {
    ETrainsFilterType,
    ITrainsFilters,
} from 'types/trains/search/filters/ITrainsFilters';
import {TTrainsFilterTypeAndValue} from 'types/trains/search/filters/TTrainsFilterTypeAndValue';
import {
    ITrainsSearchSort,
    TRAINS_SEARCH_SORT_TYPES,
    TrainsSearchSortField,
} from 'projects/trains/lib/sort/types';
import {ESortDirection} from 'types/common/sort/ESortDirection';

import getFilterIsActive from 'projects/trains/lib/filters/getFilterIsActive';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nCommonFiltersBlock from 'i18n/common-filters';
import * as i18nTrainsFiltersBlock from 'i18n/trains-filters';

import FiltersAndSortsMobile from 'components/FiltersAndSortsMobile/FiltersAndSortsMobile';
import Raskrivashka from 'components/Raskrivashka/Raskrivashka';
import Separator from 'components/Separator/Separator';
import Intersperse from 'components/Intersperse/Intersperse';
import SortMobile from 'components/SortMobile/SortMobile';
import ButtonSkeleton from 'components/Skeletons/ButtonSkeleton/ButtonSkeleton';
import TrainsTimeFilter from '../components/TrainsTimeFilter/TrainsTimeFilter';
import TrainsPriceFilter from '../components/TrainsPriceFilter/TrainsPriceFilter';
import TrainsStationFilter from '../components/TrainsStationFilter/TrainsStationFilter';
import TrainsTariffClassFilter from '../components/TrainsTariffClassFilter/TrainsTariffClassFilter';
import TrainsHighSpeedTrainFilter from '../components/TrainsHighSpeedTrainFilter/TrainsHighSpeedTrainFilter';

import cx from './TrainsSearchPageMobileFilters.scss';

interface ITrainsSearchPageMobileFiltersProps extends IWithQaAttributes {
    filters: ITrainsFilters | null;
    variantsCount: number;
    sort: ITrainsSearchSort;
    canReset: boolean;
    isLoading: boolean;

    onReset(): void;
    onChangeFilterValue(typeAndValue: TTrainsFilterTypeAndValue): void;
    onChangeSorting(sort: ITrainsSearchSort): void;
}

const TrainsSearchPageMobileFilters: React.FC<ITrainsSearchPageMobileFiltersProps> =
    props => {
        const {
            sort,
            filters,
            variantsCount,
            canReset,
            isLoading,
            onReset,
            onChangeSorting,
            onChangeFilterValue,
        } = props;

        const activeFiltersCount = useMemo(() => {
            if (!filters) {
                return 0;
            }

            return Object.values(filters).filter(getFilterIsActive).length;
        }, [filters]);

        const handleChangeSorting = useCallback(
            (type: TrainsSearchSortField, direction: ESortDirection): void => {
                onChangeSorting({
                    by: type,
                    isReverse: direction === ESortDirection.DESC,
                });
            },
            [onChangeSorting],
        );

        const handleReset = useCallback((): void => {
            handleChangeSorting(
                TrainsSearchSortField.DEPARTURE,
                ESortDirection.ASC,
            );

            onReset();
        }, [handleChangeSorting, onReset]);

        const skeleton = useMemo(() => {
            if (!isLoading) {
                return null;
            }

            return (
                <ButtonSkeleton
                    className={cx('skeleton')}
                    size="xl"
                    withAnimation
                />
            );
        }, [isLoading]);

        let filtersNode: React.ReactNode = null;

        if (filters) {
            const {
                [ETrainsFilterType.ARRIVAL]: arrivalFilter,
                [ETrainsFilterType.DEPARTURE]: departureFilter,
                [ETrainsFilterType.HIGH_SPEED_TRAIN]: highSpeedTrainFilter,
                [ETrainsFilterType.PRICE_RANGE]: priceRangeFilter,
                [ETrainsFilterType.STATION_FROM]: stationFromFilter,
                [ETrainsFilterType.STATION_TO]: stationToFilter,
                [ETrainsFilterType.TRAIN_TARIFF_CLASS]: trainTariffClassFilter,
            } = filters;

            filtersNode = (
                <>
                    {priceRangeFilter.availableWithOptions && (
                        <div className={cx('price')}>
                            <TrainsPriceFilter
                                filter={priceRangeFilter}
                                onChange={onChangeFilterValue}
                            />
                        </div>
                    )}
                    {trainTariffClassFilter.availableWithOptions && (
                        <Raskrivashka
                            contentIsInvolved={getFilterIsActive(
                                trainTariffClassFilter,
                            )}
                            label={i18nTrainsFiltersBlock.titleDashTrainTariffClassDashRedesign()}
                            withPrerender={false}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'trainTariffClass',
                                key: 'toggler',
                            })}
                        >
                            <TrainsTariffClassFilter
                                className={cx('modalItem')}
                                filter={trainTariffClassFilter}
                                onChange={onChangeFilterValue}
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: 'trainTariffClass',
                                })}
                            />
                        </Raskrivashka>
                    )}
                    {highSpeedTrainFilter.availableWithOptions && (
                        <Raskrivashka
                            contentIsInvolved={getFilterIsActive(
                                highSpeedTrainFilter,
                            )}
                            label={i18nTrainsFiltersBlock.titleDashHighSpeedTrain()}
                            withPrerender={false}
                        >
                            <TrainsHighSpeedTrainFilter
                                className={cx('modalItem')}
                                filter={highSpeedTrainFilter}
                                onChange={onChangeFilterValue}
                            />
                        </Raskrivashka>
                    )}
                    {departureFilter.availableWithOptions && (
                        <Raskrivashka
                            contentIsInvolved={getFilterIsActive(
                                departureFilter,
                            )}
                            label={i18nCommonFiltersBlock.departureTime()}
                            withPrerender={false}
                        >
                            <TrainsTimeFilter
                                className={cx('modalItem')}
                                filter={departureFilter}
                                onChange={onChangeFilterValue}
                            />
                        </Raskrivashka>
                    )}
                    {arrivalFilter.availableWithOptions && (
                        <Raskrivashka
                            contentIsInvolved={getFilterIsActive(arrivalFilter)}
                            label={i18nCommonFiltersBlock.arrivalTime()}
                            withPrerender={false}
                        >
                            <TrainsTimeFilter
                                className={cx('modalItem')}
                                filter={arrivalFilter}
                                onChange={onChangeFilterValue}
                            />
                        </Raskrivashka>
                    )}
                    {stationFromFilter.availableWithOptions && (
                        <Raskrivashka
                            contentIsInvolved={getFilterIsActive(
                                stationFromFilter,
                            )}
                            label={i18nTrainsFiltersBlock.titleDashStationFrom()}
                            withPrerender={false}
                        >
                            <TrainsStationFilter
                                className={cx('modalItem')}
                                filter={stationFromFilter}
                                onChange={onChangeFilterValue}
                            />
                        </Raskrivashka>
                    )}
                    {stationToFilter.availableWithOptions && (
                        <Raskrivashka
                            contentIsInvolved={getFilterIsActive(
                                stationToFilter,
                            )}
                            label={i18nTrainsFiltersBlock.titleDashStationTo()}
                            withPrerender={false}
                        >
                            <TrainsStationFilter
                                className={cx('modalItem')}
                                filter={stationToFilter}
                                onChange={onChangeFilterValue}
                            />
                        </Raskrivashka>
                    )}
                </>
            );
        }

        const sortChanged =
            sort.by !== TrainsSearchSortField.DEPARTURE || sort.isReverse;

        return (
            <FiltersAndSortsMobile
                className={cx('button')}
                activeFiltersCount={activeFiltersCount}
                itemsCount={variantsCount}
                isCompact
                skeleton={skeleton}
                canReset={canReset || sortChanged}
                onReset={handleReset}
                {...prepareQaAttributes(props)}
            >
                <Intersperse separator={<Separator />}>
                    {filtersNode}

                    <SortMobile
                        className={cx('modalItem')}
                        type={sort.by}
                        sortOrder={TRAINS_SEARCH_SORT_TYPES}
                        direction={
                            sort.isReverse
                                ? ESortDirection.DESC
                                : ESortDirection.ASC
                        }
                        above={5}
                        below={8}
                        onChange={handleChangeSorting}
                    />
                </Intersperse>
            </FiltersAndSortsMobile>
        );
    };

export default TrainsSearchPageMobileFilters;
