import React, {useCallback, useMemo, ReactNode} from 'react';

import {
    ETrainsFilterType,
    ITrainsSimpleFilterOption,
    ITrainsHighSpeedTrainFilter,
} from 'types/trains/search/filters/ITrainsFilters';
import {IWithClassName} from 'types/withClassName';
import {TTrainsFilterTypeAndValue} from 'types/trains/search/filters/TTrainsFilterTypeAndValue';

import toggleValue from 'projects/trains/pages/TrainsSearchPage/components/TrainsSearchPageFilters/utilities/toggleTrainsFilterValue';
import {
    convertFilterOptionToText,
    convertFilterOptionToValue,
} from 'projects/trains/pages/TrainsSearchPage/components/TrainsSearchPageFilters/utilities/convertTrainSimpleFilterOption';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import FilterCheckList from 'components/FilterCheckList/FilterCheckList';

import TrainsFilterOptionPrice from '../TrainsFilterOptionPrice/TrainsFilterOptionPrice';

interface ITrainsHighSpeedTrainFilterProps extends IWithClassName {
    filter: ITrainsHighSpeedTrainFilter;
    onChange(typeAndValue: TTrainsFilterTypeAndValue): void;
}

const TrainsHighSpeedTrainFilter: React.FC<ITrainsHighSpeedTrainFilterProps> =
    ({className, filter, onChange}) => {
        const deviceType = useDeviceType();
        const {
            value: filterValue,
            options: filterOptions,
            activeOptions: filterActiveOptions,
            optionMinPriceList: filterOptionMinPriceList,
        } = filter;
        const handleChange = useCallback(
            (option: ITrainsSimpleFilterOption) => {
                const toggledValue = toggleValue(filterValue, option.value);

                onChange({
                    type: ETrainsFilterType.HIGH_SPEED_TRAIN,
                    value: toggledValue,
                });
            },
            [filterValue, onChange],
        );

        const filterAvailableValues = useMemo(() => {
            return filterActiveOptions.map(option => option.value);
        }, [filterActiveOptions]);

        const renderOptionAside = useCallback(
            (option: ITrainsSimpleFilterOption, index): ReactNode => {
                const optionMinPrice = filterOptionMinPriceList?.[index];

                if (!optionMinPrice) {
                    return null;
                }

                return <TrainsFilterOptionPrice minPrice={optionMinPrice} />;
            },
            [filterOptionMinPriceList],
        );

        return (
            <FilterCheckList<ITrainsSimpleFilterOption, string>
                className={className}
                deviceType={deviceType}
                values={filterValue}
                options={filterOptions}
                availableValues={filterAvailableValues}
                renderText={convertFilterOptionToText}
                optionToValue={convertFilterOptionToValue}
                renderAside={renderOptionAside}
                onChange={handleChange}
            />
        );
    };

export default TrainsHighSpeedTrainFilter;
