import React, {useCallback, useMemo, ReactNode} from 'react';

import {TTrainsFilterTypeAndValue} from 'types/trains/search/filters/TTrainsFilterTypeAndValue';
import {
    ITrainsSimpleFilterOption,
    ITrainsStationFromFilter,
    ITrainsStationToFilter,
} from 'types/trains/search/filters/ITrainsFilters';
import {IWithClassName} from 'types/withClassName';

import {
    convertFilterOptionToText,
    convertFilterOptionToValue,
} from 'projects/trains/pages/TrainsSearchPage/components/TrainsSearchPageFilters/utilities/convertTrainSimpleFilterOption';
import toggleValue from 'projects/trains/pages/TrainsSearchPage/components/TrainsSearchPageFilters/utilities/toggleTrainsFilterValue';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import FilterCheckList from 'components/FilterCheckList/FilterCheckList';

import TrainsFilterOptionPrice from '../TrainsFilterOptionPrice/TrainsFilterOptionPrice';

interface ITrainsStationFilterProps extends IWithClassName {
    filter: ITrainsStationFromFilter | ITrainsStationToFilter;
    onChange: (typeAndValue: TTrainsFilterTypeAndValue) => void;
}

const TrainsStationFilter: React.FC<ITrainsStationFilterProps> = ({
    className,
    filter,
    onChange,
}) => {
    const deviceType = useDeviceType();
    const {
        type: filterType,
        value: filterValue,
        options: filterOptions,
        activeOptions: filterActiveOptions,
        optionMinPriceList: filterOptionMinPriceList,
    } = filter;
    const handleChange = useCallback(
        (option: ITrainsSimpleFilterOption) => {
            const toggledValue = toggleValue(filterValue, option.value);

            onChange({
                type: filterType,
                value: toggledValue,
            });
        },
        [filterValue, filterType, onChange],
    );

    const filterAvailableValues = useMemo(() => {
        return filterActiveOptions.map(option => option.value);
    }, [filterActiveOptions]);

    const renderOptionAside = useCallback(
        (option: ITrainsSimpleFilterOption, index): ReactNode => {
            const optionMinPrice = filterOptionMinPriceList?.[index];

            if (!optionMinPrice) {
                return null;
            }

            return <TrainsFilterOptionPrice minPrice={optionMinPrice} />;
        },
        [filterOptionMinPriceList],
    );

    return (
        <FilterCheckList<ITrainsSimpleFilterOption, string>
            className={className}
            deviceType={deviceType}
            values={filterValue}
            options={filterOptions}
            availableValues={filterAvailableValues}
            renderText={convertFilterOptionToText}
            optionToValue={convertFilterOptionToValue}
            renderAside={renderOptionAside}
            onChange={handleChange}
        />
    );
};

export default TrainsStationFilter;
