import React, {useCallback, ReactNode} from 'react';

import {
    ETrainsFilterType,
    ITrainsTrainTariffClassFilter,
} from 'types/trains/search/filters/ITrainsFilters';
import {TTrainsFilterTypeAndValue} from 'types/trains/search/filters/TTrainsFilterTypeAndValue';
import {IWithClassName} from 'types/withClassName';
import {ETrainsCoachType} from 'types/trains/common/tariffs/ETrainsCoachType';

import toggleValue from 'projects/trains/pages/TrainsSearchPage/components/TrainsSearchPageFilters/utilities/toggleTrainsFilterValue';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import getTariffClass from 'projects/trains/lib/getTariffClass';
import capitalizeFirstLetter from 'utilities/strings/capitalizeFirstLetter';

import FilterCheckList from 'components/FilterCheckList/FilterCheckList';

import TrainsFilterOptionPrice from '../TrainsFilterOptionPrice/TrainsFilterOptionPrice';

interface ITrainsTariffClassFilterProps
    extends IWithClassName,
        IWithQaAttributes {
    filter: ITrainsTrainTariffClassFilter;
    onChange(typeAndValue: TTrainsFilterTypeAndValue): void;
}

const convertOptionToValue = (option: ETrainsCoachType): ETrainsCoachType =>
    option;

const convertOptionToText = (option: ETrainsCoachType): string => {
    const tariffClassText = getTariffClass(option);

    return capitalizeFirstLetter(tariffClassText);
};

const TrainsTariffClassFilter: React.FC<ITrainsTariffClassFilterProps> = ({
    className,
    filter,
    onChange,
    ...rest
}) => {
    const deviceType = useDeviceType();
    const {
        value: filterValue,
        options: filterOptions,
        activeOptions: filterActiveOptions,
        optionMinPriceList: filterOptionMinPriceList,
    } = filter;

    const handleChange = useCallback(
        (value: ETrainsCoachType) => {
            const toggledValue = toggleValue(filterValue, value);

            onChange({
                type: ETrainsFilterType.TRAIN_TARIFF_CLASS,
                value: toggledValue,
            });
        },
        [filterValue, onChange],
    );

    const renderOptionAside = useCallback(
        (option: ETrainsCoachType, index): ReactNode => {
            const optionMinPrice = filterOptionMinPriceList?.[index];

            if (!optionMinPrice) {
                return null;
            }

            return <TrainsFilterOptionPrice minPrice={optionMinPrice} />;
        },
        [filterOptionMinPriceList],
    );

    return (
        <FilterCheckList<ETrainsCoachType, ETrainsCoachType>
            className={className}
            deviceType={deviceType}
            values={filterValue}
            options={filterOptions}
            optionToValue={convertOptionToValue}
            availableValues={filterActiveOptions}
            renderText={convertOptionToText}
            renderAside={renderOptionAside}
            onChange={handleChange}
            {...prepareQaAttributes(rest)}
        />
    );
};

export default TrainsTariffClassFilter;
