import React, {useCallback, ReactNode} from 'react';

import {PERIOD_MAP} from './utilities/constants';
import {TIMES_OF_DAY} from 'utilities/dateUtils/constants';

import {ETimeOfDay} from 'utilities/dateUtils/types';
import {TTrainsFilterTypeAndValue} from 'types/trains/search/filters/TTrainsFilterTypeAndValue';
import {
    ETrainsFilterType,
    ITrainsArrivalFilter,
    ITrainsDepartureFilter,
} from 'types/trains/search/filters/ITrainsFilters';
import {IWithClassName} from 'types/withClassName';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import toggleValue from 'projects/trains/pages/TrainsSearchPage/components/TrainsSearchPageFilters/utilities/toggleTrainsFilterValue';

import CommonTimeFilter from 'components/TimeFilter/TimeFilter';

import TrainsFilterOptionPrice from '../TrainsFilterOptionPrice/TrainsFilterOptionPrice';

interface ITrainsTimeFilterProps extends IWithClassName {
    title?: string;
    filter: ITrainsArrivalFilter | ITrainsDepartureFilter;
    onChange: (typeAndValue: TTrainsFilterTypeAndValue) => void;
}

const TrainsTimeFilter: React.FC<ITrainsTimeFilterProps> = ({
    className,
    filter,
    title,
    onChange,
}) => {
    const deviceType = useDeviceType();
    const {
        type: filterType,
        value: filterValue,
        activeOptions: filterActiveOptions,
        optionMinPriceList: filterOptionMinPriceList,
    } = filter;

    const handleChange = useCallback(
        (value: ETimeOfDay) => {
            const toggledValue = toggleValue(filterValue, value);

            onChange({
                type: filterType,
                value: toggledValue,
            });
        },
        [filterValue, filterType, onChange],
    );
    const renderOptionAside = useCallback(
        (option: ETimeOfDay, index): ReactNode => {
            const optionMinPrice = filterOptionMinPriceList?.[index];

            if (!optionMinPrice) {
                return null;
            }

            return <TrainsFilterOptionPrice minPrice={optionMinPrice} />;
        },
        [filterOptionMinPriceList],
    );

    return (
        <CommonTimeFilter
            className={className}
            deviceType={deviceType}
            title={title}
            value={filterValue}
            availableValues={filterActiveOptions}
            options={TIMES_OF_DAY}
            periodMap={
                filterType === ETrainsFilterType.DEPARTURE
                    ? PERIOD_MAP
                    : undefined
            }
            renderPrice={renderOptionAside}
            onChange={handleChange}
        />
    );
};

export default TrainsTimeFilter;
