import {useCallback, useEffect, FC} from 'react';

import {EDirection} from 'types/common/EDirection';
import {ETrainsGoal} from 'utilities/metrika/types/goals/trains';
import {ITrainsFilledSearchContext} from 'reducers/trains/context/types';
import {INearestTrainDatesByDirection} from 'types/trains/search/searchInfo/INearestTrainDates';

import {reachGoal} from 'utilities/metrika';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {getNearestTrainDatesTitleByReason} from './utilities/getNearestTrainDatesTitleByReason';
import {getNearestTrainDatesByDirection} from 'projects/trains/lib/genericSearch/nearestTrainDates/getNearestTrainDatesByDirection';
import {checkHasNearestTrainDatesByDirection} from 'projects/trains/lib/genericSearch/nearestTrainDates/checkHasNearestTrainDatesByDirection';

import * as i18nBlock from 'i18n/trains-NearestTrainDates';

import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import Box, {TBoxSizes} from 'components/Box/Box';
import TrainsSearchPageNearestTrainDate from './components/TrainsSearchPageNearestTrainDate/TrainsSearchPageNearestTrainDate';

import cx from './TrainsSearchPageNearestTrainDates.scss';

interface ITrainsSearchPageNearestTrainDatesProps {
    context: ITrainsFilledSearchContext;
    belowContainer?: TBoxSizes;
    aboveContainer?: TBoxSizes;
    direction?: EDirection | null;
    nearestTrainDatesByDirection?: INearestTrainDatesByDirection;
}

const TrainsSearchPageNearestTrainDates: FC<ITrainsSearchPageNearestTrainDatesProps> =
    props => {
        const {
            context,
            direction,
            aboveContainer,
            belowContainer,
            nearestTrainDatesByDirection,
        } = props;
        const deviceType = useDeviceType();

        const hasNearestTrainDatesByDirection =
            checkHasNearestTrainDatesByDirection({
                direction,
                nearestTrainDatesByDirection,
                isRoundTrip: Boolean(context.returnWhen),
            });
        const nearestTrainDates = getNearestTrainDatesByDirection({
            direction,
            nearestTrainDatesByDirection,
        });

        const handleClickDateIntent = useCallback(() => {
            reachGoal(ETrainsGoal.SEARCH_NEAREST_TRAIN_DATES_CLICK);
        }, []);

        useEffect(() => {
            if (hasNearestTrainDatesByDirection) {
                reachGoal(ETrainsGoal.SEARCH_RENDER_NEAREST_TRAIN_DATES);
            }
        }, [hasNearestTrainDatesByDirection]);

        if (
            !hasNearestTrainDatesByDirection ||
            !nearestTrainDates?.dates?.length
        ) {
            return null;
        }

        return (
            <Flex
                className={cx('root')}
                flexDirection="column"
                between={3}
                above={aboveContainer}
                below={belowContainer}
            >
                <Box between={deviceType.isMobile ? 1 : 2}>
                    <Text
                        size={deviceType.isMobile ? 'l' : 'xl'}
                        weight="bold"
                        color="primary"
                        tag="div"
                    >
                        {getNearestTrainDatesTitleByReason({
                            reason: nearestTrainDates.reason,
                        })}
                    </Text>
                    <Text size="m" color="primary" tag="div">
                        {i18nBlock.nearestTrainDates()}
                    </Text>
                </Box>
                <Flex flexWrap="wrap" between={2} inline>
                    {nearestTrainDates.dates.map(({date}) => {
                        return (
                            <TrainsSearchPageNearestTrainDate
                                className={cx('intent')}
                                deviceType={deviceType}
                                key={date}
                                date={date}
                                context={context}
                                onClick={handleClickDateIntent}
                            />
                        );
                    })}
                </Flex>
            </Flex>
        );
    };

export default TrainsSearchPageNearestTrainDates;
