import React from 'react';

import {EDirection} from 'types/common/EDirection';
import {ITrainsVariant} from 'types/trains/common/variant/ITrainsVariant';
import {EQueryingStatus} from 'types/trains/search/searchInfo/ITrainsSearchInfo';
import {ITrainsFilledSearchContext} from 'reducers/trains/context/types';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {getTrainsVariantFirstSegment} from 'projects/trains/lib/genericSearch/variants/getTrainsVariantFirstSegment';
import {checkTrainsHasPinnedSearchSegment} from 'projects/trains/lib/genericSearch/segments/checkTrainsHasPinnedSearchSegment';

import * as i18nBlock from 'i18n/trains-SearchPinnedSegment';

import Text from 'components/Text/Text';
import CardWithDeviceLayout from 'components/CardWithDeviceLayout/CardWithDeviceLayout';
import TrainsSearchSegment from 'projects/trains/components/TrainsSearchVariants/components/TrainsSearchSegment/TrainsSearchSegment';

import cx from './TrainsSearchPagePinnedSegment.scss';

interface ITrainsSearchPagePinnedSegmentProps {
    variants: ITrainsVariant[];
    searchStatus: EQueryingStatus;
    context: ITrainsFilledSearchContext;
}

const TrainsSearchPagePinnedSegment: React.FC<ITrainsSearchPagePinnedSegmentProps> =
    props => {
        const deviceType = useDeviceType();
        const {variants, context, searchStatus} = props;
        const {direction} = context;
        const [firstVariant] = variants ?? [];
        const hasPinnedSearchSegment = checkTrainsHasPinnedSearchSegment({
            context,
        });

        if (!firstVariant || !hasPinnedSearchSegment) {
            return null;
        }

        const isBackwardDirection = direction === EDirection.BACKWARD;
        const pinnedSegmentDirection = isBackwardDirection
            ? EDirection.FORWARD
            : EDirection.BACKWARD;
        const pinnedSegment = getTrainsVariantFirstSegment({
            variant: firstVariant,
            direction: pinnedSegmentDirection,
        });

        if (!pinnedSegment) {
            return null;
        }

        const segmentNode = (
            <TrainsSearchSegment
                canHidePrice
                isPinnedSegment
                segment={pinnedSegment}
                context={context}
                variant={firstVariant}
                searchStatus={searchStatus}
                direction={pinnedSegmentDirection}
            />
        );

        return (
            <div className={cx('root', deviceMods('root', deviceType))}>
                <Text
                    className={cx('title')}
                    tag="div"
                    color="primary"
                    size={deviceType.isMobile ? 'l' : 'xl'}
                >
                    {isBackwardDirection
                        ? i18nBlock.forwardTitle()
                        : i18nBlock.backwardTitle()}
                </Text>
                {deviceType.isMobile ? (
                    segmentNode
                ) : (
                    <CardWithDeviceLayout>{segmentNode}</CardWithDeviceLayout>
                )}
            </div>
        );
    };

export default TrainsSearchPagePinnedSegment;
