import React from 'react';
import {times} from 'lodash';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {TrainsSearchContextType} from 'reducers/trains/context/types';

import {deviceMods} from 'utilities/stylesUtils';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {checkTrainsHasPinnedSearchSegment} from 'projects/trains/lib/genericSearch/segments/checkTrainsHasPinnedSearchSegment';

import Box from 'components/Box/Box';
import TextSkeleton from 'components/Skeletons/TextSkeleton/TextSkeleton';

import TrainsSearchPageSegmentSkeleton from './TrainsSearchPageSegmentSkeleton/TrainsSearchPageSegmentSkeleton';

import cx from './TrainsSearchPagePreloader.scss';

interface ITrainsSearchPagePreloaderProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    context: TrainsSearchContextType;
}

const SKELETONS_DESKTOP_COUNT = 5;
const SKELETONS_MOBILE_COUNT = 4;

const TrainsSearchPagePreloader: React.FC<ITrainsSearchPagePreloaderProps> =
    props => {
        const {className, deviceType, context} = props;
        const hasPinnedSearchSegment = checkTrainsHasPinnedSearchSegment({
            context,
        });

        return (
            <div
                className={cx(
                    'root',
                    className,
                    deviceMods('root', deviceType),
                )}
                {...prepareQaAttributes(props)}
            >
                {hasPinnedSearchSegment && (
                    <div className={cx('pinnedSegment')}>
                        <TextSkeleton
                            className={cx('pinnedSegmentTitle')}
                            withAnimation
                            size={deviceType.isMobile ? 'l' : 'xl'}
                        />
                        <TrainsSearchPageSegmentSkeleton
                            deviceType={deviceType}
                        />
                    </div>
                )}
                {(deviceType.isDesktop ||
                    (deviceType.isMobile && context.returnWhen)) && (
                    <TextSkeleton
                        className={cx('title')}
                        withAnimation
                        size={deviceType.isMobile ? 'l' : 'xl'}
                    />
                )}

                {deviceType.isDesktop && (
                    <TextSkeleton
                        className={cx('subtitle')}
                        withAnimation
                        size={deviceType.isMobile ? 'xl' : 's'}
                    />
                )}

                <Box className={cx('segments')} between="3">
                    {times(
                        deviceType.isMobile
                            ? SKELETONS_MOBILE_COUNT
                            : SKELETONS_DESKTOP_COUNT,
                    ).map(index => (
                        <TrainsSearchPageSegmentSkeleton
                            key={index}
                            deviceType={deviceType}
                        />
                    ))}
                </Box>

                <div className={cx('overlay')} />
            </div>
        );
    };

export default TrainsSearchPagePreloader;
