import React, {useMemo} from 'react';

import {IWithClassName} from 'types/withClassName';
import {ITrainsFilledSearchContext} from 'reducers/trains/context/types';
import {ITrainsVariant} from 'types/trains/common/variant/ITrainsVariant';
import {INearestTrainDatesByDirection} from 'types/trains/search/searchInfo/INearestTrainDates';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import IPrice from 'utilities/currency/PriceInterface';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useExperiments} from 'utilities/hooks/useExperiments';
import {getTrainsSearchPageTitle} from './utilities/getTrainsSearchPageTitle';
import {getTrainsVariantsMinPrice} from 'projects/trains/lib/genericSearch/variants/getTrainsVariantsMinPrice';
import {checkHasNearestTrainDatesByContext} from 'projects/trains/lib/genericSearch/nearestTrainDates/checkHasNearestTrainDatesByContext';

import Price from 'components/Price/Price';
import Heading from 'components/Heading/Heading';

import cx from './TrainsSearchPageTitle.scss';

interface ITrainsSearchPageTitleProps
    extends IWithClassName,
        IWithQaAttributes {
    context: ITrainsFilledSearchContext;
    variants: ITrainsVariant[];
    nearestTrainDatesByDirection?: INearestTrainDatesByDirection;
}

const TrainsSearchPageTitle: React.FC<ITrainsSearchPageTitleProps> = props => {
    const {className, context, variants, nearestTrainDatesByDirection} = props;
    const {direction, returnWhen} = context;
    const deviceType = useDeviceType();
    const {trainsNearestTrainDates: trainsNearestTrainDatesEnabled} =
        useExperiments();
    const hasNearestTrainDates =
        trainsNearestTrainDatesEnabled &&
        checkHasNearestTrainDatesByContext({
            searchContext: context,
            nearestTrainDatesByDirection,
        });
    const title = getTrainsSearchPageTitle({context, hasNearestTrainDates});
    const variantsMinPrice = useMemo<IPrice | undefined>(
        () => getTrainsVariantsMinPrice({variants, direction}),
        [variants, direction],
    );
    const canRenderVariantMinPrice =
        variantsMinPrice && !returnWhen && !hasNearestTrainDates;

    return (
        <header className={className} {...prepareQaAttributes(props)}>
            <Heading
                className={cx('title', deviceMods('title', deviceType))}
                level="1"
            >
                {title}
            </Heading>
            {canRenderVariantMinPrice && (
                <Price
                    className={cx('price', deviceMods('price', deviceType))}
                    isFrom
                    isRound
                    value={variantsMinPrice?.value}
                    currency={variantsMinPrice?.currency}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'price',
                    })}
                />
            )}
        </header>
    );
};

export default TrainsSearchPageTitle;
