import React, {useMemo} from 'react';

import {TROUBLESHOOTING_FORM_URL} from 'constants/common';
import {URLs} from 'constants/urls';

import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import handleRefreshClick from './utilities/handleRefreshClick';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/trips-common';

import Error500Image from 'icons/illustrations/Error500/Error500M';
import SupportPhone from 'components/SupportPhone/SupportPhone';
import ErrorModal, {TErrorActionType} from 'components/ErrorModal/ErrorModal';
import Link from 'components/Link/Link';

import {usePlatform} from 'contexts/PlatformContext';

import cx from './LoadingErrorModal.scss';

interface ILoadingErrorModalProps extends IWithQaAttributes {
    isVisible: boolean;
    onNavigateToIndex?(): void;
    onRefreshClick?(): void;
}

const ERROR_IMAGE = <Error500Image className={cx('image')} />;

const LoadingErrorModal: React.FC<ILoadingErrorModalProps> = props => {
    const {
        isVisible,
        onNavigateToIndex,
        onRefreshClick = handleRefreshClick,
    } = props;

    const {isWeb} = usePlatform();

    const primaryAction = useMemo((): TErrorActionType => {
        return {
            type: 'button',
            title: i18nBlock.errorRefreshPage(),
            handler: onRefreshClick,
        };
    }, [onRefreshClick]);

    const secondaryAction = useMemo((): TErrorActionType => {
        return {
            type: 'button-link',
            title: i18nBlock.errorToIndex(),
            props: {
                to: URLs.index,
            },
            handler: onNavigateToIndex,
        };
    }, [onNavigateToIndex]);

    return (
        <ErrorModal
            isVisible={isVisible}
            title={i18nBlock.errorTitle()}
            text={
                isWeb
                    ? insertJSXIntoKey(i18nBlock.errorDescription)({
                          phone: <SupportPhone />,
                          link: (
                              <Link
                                  url={TROUBLESHOOTING_FORM_URL}
                                  target="_blank"
                                  rel="noopener noreferrer"
                              >
                                  {i18nBlock.errorFeedback()}
                              </Link>
                          ),
                      })
                    : ''
            }
            image={ERROR_IMAGE}
            primaryAction={primaryAction}
            secondaryAction={isWeb ? secondaryAction : undefined}
            {...prepareQaAttributes(props)}
        />
    );
};

export default React.memo(LoadingErrorModal);
