import React, {useCallback} from 'react';

import {IWithClassName} from 'types/withClassName';
import ETripImageType from './types/ETripImageType';

import getStubComponent from './utilities/getStubComponent';
import {useBoolean} from 'utilities/hooks/useBoolean';
import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import TravelImage from 'components/TravelImage/TravelImage';

import cx from './TripImage.scss';

interface ITripImageProps extends IWithClassName, IWithQaAttributes {
    src: string | undefined;
    type?: ETripImageType;
    enableFog?: boolean;
    /**
     * Анимация зума при наведении (только десктоп)
     */
    enableZoom?: boolean;
    /**
     * Использовать картинку заглушку из TravelImage
     */
    useDefaultTravelImageStub?: boolean;
    /**
     * Делает картинку черно-белой и прозрачной
     */
    enableGray?: boolean;
    onError?(): void;
}

const TripImage: React.FC<ITripImageProps> = props => {
    const {
        className,
        src,
        type = ETripImageType.SMALL_CIRCLE,
        enableFog = false,
        enableZoom = false,
        enableGray = false,
        useDefaultTravelImageStub = false,
        onError,
    } = props;

    const {setTrue: setIsErrorTrue, value: isError} = useBoolean(false);
    const handleError = useCallback(() => {
        setIsErrorTrue();
        onError?.();
    }, [setIsErrorTrue, onError]);

    const deviceType = useDeviceType();

    const StubComponent = getStubComponent(type, deviceType);

    return (
        <div
            className={cx(
                className,
                'root',
                `root_${type}`,
                deviceMods('root', deviceType),
                {
                    root_zoom: deviceType.isDesktop && enableZoom,
                    root_gray: enableGray,
                },
            )}
            {...prepareQaAttributes(props)}
        >
            {(src && !isError) || useDefaultTravelImageStub ? (
                <TravelImage
                    className={cx('travelImage')}
                    imageClassName={cx('image')}
                    src={src}
                    onError={handleError}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'image',
                    })}
                />
            ) : (
                <div
                    className={cx('stubContainer')}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'stubContainer',
                    })}
                >
                    <StubComponent className={cx('stub')} />
                </div>
            )}
            {enableFog && <div className={cx('fog')} />}
        </div>
    );
};

export default TripImage;
