import React, {useEffect, useMemo} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {EProjectName} from 'constants/common';
import {PROJECT_FOOTER_TYPES} from 'components/Footer/constants/footerMenuItems';

import {ELoadableReducer} from 'types/common/ELoadableReducer';
import {EAdFoxBannerPosition, EAdFoxBannerType} from 'types/AdFox';
import {ECommonGoal} from 'utilities/metrika/types/goals/common';

import searchSuggestsReducer from 'reducers/hotels/searchSuggests/reducer';
import searchInformationReducer from 'reducers/hotels/searchInformation/reducer';
import {fetchCrossLinks} from 'reducers/common/crossLinks/thunk';
import crossLinksReducer from 'reducers/common/crossLinks/reducer';

import crossLinksSelector from 'selectors/common/crossLinksSelector';
import nonceSelector from 'selectors/common/nonceSelector';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import prepareCrossLinksForGallery from 'utilities/prepareCrossLinksForGallery';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';

import * as i18nBlock from 'i18n/hotels-Recipes';

import withReducers from 'containers/withReducers/withReducers';

import Text from 'components/Text/Text';
import CrossLinksGallery from 'projects/index/IndexApp/components/CrossLinksGallery/CrossLinksGallery';
import LayoutDefault from 'components/Layouts/LayoutDefault/LayoutDefault';
import WhiteLabelSearchFormBlock from './components/WhiteLabelSearchFormBlock/WhiteLabelSearchFormBlock';
import WhiteLabelAdvantages from './components/WhiteLabelAdvantages/WhiteLabelAdvantages';
import AdFoxBanner from 'components/AdFoxBanner/AdFoxBanner';

import cx from './WhiteLabelApp.scss';

/**
 * Конфиги партнеров импортируются для того чтобы вебпак разруливал
 * ссылки на статику для клиента. Нужно добавлять импорт конфигурации
 * каждого партнера
 **/
import './configs/s7.config';

const ROOT_QA = 'indexWhiteLabelPage';

const WhiteLabelApp: React.FC = () => {
    const deviceType = useDeviceType();
    const dispatch = useDispatch();
    const nonce = useSelector(nonceSelector);

    const footerType = PROJECT_FOOTER_TYPES[EProjectName.WHITE_LABEL];

    const crossLinksState = useSelector(crossLinksSelector);
    const {project, isLoading, isFetched} = crossLinksState;

    const renderCrossaleBlock = project === EProjectName.HOTELS;

    const galleryItems = useMemo(() => {
        return prepareCrossLinksForGallery(
            crossLinksState,
            EProjectName.HOTELS,
        );
    }, [crossLinksState]);

    useEffect(() => {
        if (project !== EProjectName.HOTELS || !isFetched) {
            dispatch(fetchCrossLinks(EProjectName.HOTELS));
        }
    }, [dispatch, isFetched, project]);

    const crossLinkGalleryTitle = useMemo(() => {
        return insertJSXIntoKey(i18nBlock.title)({
            start: (
                <Text weight="fat" size="inherit">
                    {i18nBlock.searchHotels()}
                </Text>
            ),
            end: (
                <Text weight="fat" size="inherit">
                    {i18nBlock.atCompetitivePrices()}
                </Text>
            ),
        });
    }, []);

    return (
        <LayoutDefault
            isIndex
            project={EProjectName.WHITE_LABEL}
            footerType={footerType}
            className={cx('destinations')}
            footerWrapperClassName={cx('footer')}
            footerWithoutSeparator
        >
            <WhiteLabelSearchFormBlock />

            <div className={cx('content', deviceMods('content', deviceType))}>
                <AdFoxBanner
                    className={cx('banner', deviceMods('banner', deviceType))}
                    fixed
                    type={EAdFoxBannerType.Inline}
                    position={EAdFoxBannerPosition.Top}
                />

                <WhiteLabelAdvantages className={cx('advantages')} />

                {renderCrossaleBlock && (
                    <CrossLinksGallery
                        className={cx('recipes')}
                        title={crossLinkGalleryTitle}
                        items={galleryItems}
                        loading={isLoading}
                        nonce={nonce}
                        markupText={i18nBlock.blockDashTitleDashNew()}
                        itemClickMetrikaGoal={ECommonGoal.RECIPE_LINK}
                        {...prepareQaAttributes({
                            parent: ROOT_QA,
                            current: 'crossLinksGallery',
                        })}
                    />
                )}
            </div>
        </LayoutDefault>
    );
};

export default withReducers([
    [ELoadableReducer.HOTELS_SEARCH_SUGGESTS, searchSuggestsReducer],
    [ELoadableReducer.HOTELS_SEARCH_INFORMATION, searchInformationReducer],
    [ELoadableReducer.COMMON_CROSSLINKS, crossLinksReducer],
])(WhiteLabelApp);
