import React, {ReactNode, useCallback} from 'react';

import {IWithClassName} from 'types/withClassName';
import IIconAdvantageProps from 'components/Advantages/types/IIconAdvantageProps';

import {
    getQa,
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useWhiteLabelConfig} from 'utilities/hooks/useWhiteLabelConfig';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import {TSwipeableRenderItemParams} from 'components/Advantages/Advantages';
import Swipeable from 'components/Swipeable/Swipeable';
import Advantage from 'components/Advantages/components/Advantage/Advantage';
import CardWithDeviceLayout from 'components/CardWithDeviceLayout/CardWithDeviceLayout';
import Flex from 'components/Flex/Flex';

import cx from './WhiteLabelAdvantages.scss';

interface IWhiteLabelAdvantages extends IWithClassName, IWithQaAttributes {}

const ICON_SIZE = 120;

const WhiteLabelAdvantages: React.FC<IWhiteLabelAdvantages> = props => {
    const {className} = props;
    const propsQa = getQa(props);

    const deviceType = useDeviceType();
    const {config: whiteLabelConfig} = useWhiteLabelConfig();
    const makeAdvantageItem = useCallback(
        ({title, description, image}) => {
            return {
                title,
                description,
                IconComponent: ({width, height}: IIconAdvantageProps) => (
                    <img
                        src={image}
                        width={width}
                        height={height}
                        alt={whiteLabelConfig?.partner}
                    />
                ),
                iconWidth: ICON_SIZE,
                iconHeight: ICON_SIZE,
            };
        },
        [whiteLabelConfig],
    );
    const advantages = (whiteLabelConfig?.advantages ?? []).map(
        makeAdvantageItem,
    );

    const renderItem = useCallback(
        ({data, index}: TSwipeableRenderItemParams): ReactNode => (
            <Advantage
                className={cx('item')}
                key={index}
                title={data.title}
                description={data.description}
                IconComponent={data.IconComponent}
                iconWidth={data.iconWidth}
                iconHeight={data.iconHeight}
                {...prepareQaAttributes({
                    parent: propsQa,
                    current: 'advantage',
                    key: index,
                })}
            />
        ),
        [propsQa],
    );

    if (advantages.length === 0) {
        return null;
    }

    return deviceType.isMobile ? (
        <CardWithDeviceLayout className={className}>
            <Swipeable
                className={cx('swiper')}
                sideThreshold={0.1}
                renderItem={renderItem}
                itemsData={advantages}
                showNav
            />
        </CardWithDeviceLayout>
    ) : (
        <Flex inline nowrap between={5} className={className}>
            {advantages.map((data, index) => (
                <CardWithDeviceLayout
                    className={cx('card')}
                    key={data.title + index}
                >
                    {renderItem({data, index})}
                </CardWithDeviceLayout>
            ))}
        </Flex>
    );
};

export default React.memo(WhiteLabelAdvantages);
