import moment, {Moment} from 'moment';

import {IDocumentDTO} from 'server/api/TravelersApi/types/IDocumentDTO';
import {IPassengerWithDocumentsDTO} from 'server/api/TravelersApi/types/IPassengerDTO';

import {IPreparedDocument} from 'reducers/account/docs/props';

import {HUMAN_DATE_RU} from 'utilities/dateUtils/formats';
import {DateLikeType, getNow} from 'utilities/dateUtils';

/**
 * @param {DateLikeType} expirationDateString - Строка даты из API (может быть пустой)
 *
 * @return {Object} {
 *     expirationDate - moment строки даты, если она была (может быть undefined)
 *     isExpired - флаг, просрочен ли документ (если нет даты - то ставим false)
 * }
 */
const prepareExpiration = (
    expirationDateString?: DateLikeType,
): {expirationDate?: string; isExpired: boolean} => {
    if (expirationDateString) {
        const expirationDate: Moment = moment(expirationDateString);

        return {
            expirationDate: expirationDate.format(HUMAN_DATE_RU),
            isExpired: expirationDate.isBefore(getNow()),
        };
    }

    return {
        isExpired: false,
    };
};

const prepareEdited = (item: IDocumentDTO): IPreparedDocument => {
    const {
        id,
        type,
        number,
        citizenship,
        passenger_id: passengerId,
        first_name: firstName,
        first_name_en: firstNameEn,
        last_name: lastName,
        last_name_en: lastNameEn,
        middle_name: middleName,
        middle_name_en: middleNameEn,
        expiration_date: expirationDate,
    } = item;

    return {
        id,
        passengerId,
        type,
        number,
        citizenship,
        firstName: firstName ? firstName : '',
        firstNameEn: firstNameEn ? firstNameEn : '',
        middleName: middleName ? middleName : '',
        middleNameEn: middleNameEn ? middleNameEn : '',
        lastName: lastName ? lastName : '',
        lastNameEn: lastNameEn ? lastNameEn : '',
        isLoading: false,
        ...prepareExpiration(expirationDate),
    };
};

/**
 *
 * @param {IDocumentDTO | IPassengerWithDocumentsDTO[]} data - может быть либо массивом пассажиров пользователя, либо отдельным документом
 *
 * @return {IPreparedDocument[] | IPreparedDocument} - возвращает либо массив подготовленных документов либо отдельный документ
 */

function prepareFetched(data: IDocumentDTO): IPreparedDocument;
function prepareFetched(
    data: IPassengerWithDocumentsDTO[],
): IPreparedDocument[];
function prepareFetched(
    data: IDocumentDTO | IPassengerWithDocumentsDTO[],
): IPreparedDocument | IPreparedDocument[];
function prepareFetched(
    data: IDocumentDTO | IPassengerWithDocumentsDTO[],
): IPreparedDocument | IPreparedDocument[] {
    const result: IPreparedDocument[] = [];

    if (Array.isArray(data)) {
        data.forEach(passenger => {
            passenger.documents.forEach(item =>
                result.push(prepareEdited(item)),
            );
        });

        return result;
    }

    return prepareEdited(data);
}

export default prepareFetched;
