import {batchActions} from 'redux-batched-actions';

import {OPERATION_ORDER_ERROR} from 'constants/errors/orderErrors';

import IRefundPartCtx from 'server/api/GenericOrderApi/types/common/refund/IRefundPartCtx';
import {EGenericOrderRefundStatus} from 'reducers/account/genericOrder/refund/status/types';
import {isNotNull} from 'types/utilities';

import {CustomThunkAction} from 'reducers/trains/customDispatch';
import {getGenericOrderRefundAmountInfoActions} from 'reducers/account/genericOrder/refund/amountInfo/actions';
import {setIsRefundModalOpenedAction} from 'reducers/account/genericOrder/refund/isModalOpened/actions';
import {setRefundStatusAction} from 'reducers/account/genericOrder/refund/status/actions';
import * as actions from 'reducers/account/orders/actions';

import genericOrderItemInfoSelector from 'selectors/account/genericOrder/item/genericOrderItemInfoSelector';
import isRefundModalOpenedSelector from 'selectors/account/genericOrder/refund/isRefundModalOpenedSelector';

import {logError} from 'utilities/logger/logError';

import {genericOrderBrowserProvider} from 'serviceProvider/genericOrder/genericOrderBrowserProvider';

export default function initRefundAction(
    refundPartContexts: IRefundPartCtx[],
): CustomThunkAction<void> {
    return async (dispatch, getState): Promise<void> => {
        try {
            const {value: genericOrder} = genericOrderItemInfoSelector(
                getState(),
            );

            if (!genericOrder) {
                return;
            }

            dispatch(
                batchActions([
                    setRefundStatusAction(
                        EGenericOrderRefundStatus.CALCULATING_AMOUNT,
                    ),
                    setIsRefundModalOpenedAction(true),
                    getGenericOrderRefundAmountInfoActions.request(
                        refundPartContexts,
                    ),
                ]),
            );

            const response =
                await genericOrderBrowserProvider.calculateRefundAmount({
                    orderId: genericOrder.id,
                    refundPartContexts,
                });

            const hasModalClosedDuringRequest = !isRefundModalOpenedSelector(
                getState(),
            );

            dispatch(
                batchActions(
                    [
                        getGenericOrderRefundAmountInfoActions.success(
                            response,
                        ),
                        hasModalClosedDuringRequest
                            ? null
                            : setRefundStatusAction(
                                  EGenericOrderRefundStatus.READY_FOR_REFUND,
                              ),
                    ].filter(isNotNull),
                ),
            );
        } catch (err) {
            dispatch(
                batchActions([
                    getGenericOrderRefundAmountInfoActions.failure(),
                    actions.orderRefund.failure(OPERATION_ORDER_ERROR),
                ]),
            );

            logError(
                {
                    message:
                        '[YATRAVEL][ACCOUNT] Generic order - подсчет суммы возврата',
                },
                err,
            );
        }
    };
}
