import {batchActions} from 'redux-batched-actions';

import {OPERATION_ORDER_ERROR} from 'constants/errors/orderErrors';
import {SECOND} from 'utilities/dateUtils/constants';

import {EGenericOrderRefundStatus} from 'reducers/account/genericOrder/refund/status/types';
import EGenericOrderState, {
    isGenericOrderState,
} from 'server/api/GenericOrderApi/types/common/EGenericOrderState';
import EGenericOrderSource from 'server/api/GenericOrderApi/types/common/EGenericOrderSource';

import {CustomThunkAction} from 'reducers/trains/customDispatch';
import {setRefundStatusAction} from 'reducers/account/genericOrder/refund/status/actions';
import {setIsRefundModalOpenedAction} from 'reducers/account/genericOrder/refund/isModalOpened/actions';
import * as actions from 'reducers/account/orders/actions';
import {getGenericOrderItemActions} from 'reducers/account/genericOrder/item/actions';

import genericOrderItemInfoSelector from 'selectors/account/genericOrder/item/genericOrderItemInfoSelector';
import refundAmountInfoSelector from 'selectors/account/genericOrder/refund/refundAmountInfoSelector';
import refundStatusSelector from 'selectors/account/genericOrder/refund/refundStatusSelector';

import {delay} from 'utilities/async/delay';
import {logError} from 'utilities/logger/logError';

import {genericOrderBrowserProvider} from 'serviceProvider/genericOrder/genericOrderBrowserProvider';

/*
 * Если возвращаем один билет и в заказе еще остались несданные билеты, то ждем статуса CONFIRMED,
 * иначе заказ должен перейти в статус REFUNDED.
 */
const AFTER_REFUND_ORDER_STATUSES = [
    EGenericOrderState.REFUNDED,
    EGenericOrderState.CONFIRMED,
];

export default function startRefundAction(): CustomThunkAction<void> {
    return async (dispatch, getState): Promise<void> => {
        try {
            const state = getState();

            const {value: refundAmountInfoContext} =
                refundAmountInfoSelector(state);

            const {value: order} = genericOrderItemInfoSelector(state);

            if (!refundAmountInfoContext || !order) {
                return;
            }

            dispatch(
                setRefundStatusAction(EGenericOrderRefundStatus.REFUNDING),
            );

            let {versionHash} = await genericOrderBrowserProvider.getOrderState(
                {
                    orderId: order.id,
                },
            );

            await genericOrderBrowserProvider.startRefund({
                refundToken: refundAmountInfoContext.refundToken,
                orderId: order.id,
            });

            const isQuerying = (): boolean =>
                refundStatusSelector(getState()) ===
                EGenericOrderRefundStatus.REFUNDING;

            while (isQuerying()) {
                await delay(2 * SECOND);

                const {state: orderState, versionHash: newVersionHash} =
                    await genericOrderBrowserProvider.getOrderState({
                        orderId: order.id,
                    });

                if (newVersionHash === versionHash) {
                    continue;
                }

                versionHash = newVersionHash;

                if (
                    !isGenericOrderState(orderState) ||
                    !AFTER_REFUND_ORDER_STATUSES.includes(orderState)
                ) {
                    continue;
                }

                const updatedOrder = await genericOrderBrowserProvider.getOrder(
                    {
                        orderId: order.id,
                        source: EGenericOrderSource.ORDER_PAGE,
                    },
                );

                dispatch(
                    batchActions([
                        setRefundStatusAction(
                            EGenericOrderRefundStatus.NOT_STARTED,
                        ),
                        setIsRefundModalOpenedAction(false),
                        getGenericOrderItemActions.success(updatedOrder),
                    ]),
                );
            }
        } catch (err) {
            dispatch(actions.orderRefund.failure(OPERATION_ORDER_ERROR));

            logError(
                {
                    message: '[YATRAVEL][ACCOUNT] Generic order - возврат',
                },
                err,
            );
        }
    };
}
