import {ActionType, createReducer} from 'typesafe-actions';

import {ORDER_TYPE_VALUES} from 'projects/account/lib/orders/types';
import {IAccountOrdersError} from 'reducers/account/orders/types/IAccountOrdersError';
import {IGenericOrderInfo} from 'server/api/GenericOrderApi/types/common/IGenericOrderInfo';
import {ORDER_TRAINS_REGISTRATION_STATUS} from 'projects/account/pages/Order/types/OrderTrainsRegistrationStatus';

import * as actions from 'reducers/account/orders/actions';
import {
    checkOrderAuthorizationActions,
    TCheckOrderAuthorizationActionsType,
} from 'reducers/common/checkOrderAuthorization/actions';

import {STATUS} from 'projects/account/lib/orders/actionsStatuses';
import IPrice from 'utilities/currency/PriceInterface';
import {convertOrderType} from 'server/api/HotelsBookAPI/utilities/prepareCheckOrderAuthorization';

export interface IOrdersState {
    selectedTickets: string[];
    trainsOrder: IGenericOrderInfo | null;
    refundAmount: IPrice | null;
    penaltyAmount: IPrice | null;
    refundToken: string | null;
    trainsOrderType:
        | ORDER_TYPE_VALUES.TRAINS
        | ORDER_TYPE_VALUES.TRAINS_GENERIC
        | null;
    isFetching: boolean;
    isFetched: boolean;
    isError: boolean;
    isAuthError: boolean;
    isRefundModalOpened: boolean;
    isRegistrationModalOpened: boolean;
    error: IAccountOrdersError | null;
    updateStatus: STATUS;
    refundStatus: STATUS;
    requestedRegistrationStatus: ORDER_TRAINS_REGISTRATION_STATUS | null;
}

const initialState: IOrdersState = {
    isFetching: false,
    isFetched: false,
    isError: false,
    error: null,
    trainsOrder: null,
    refundAmount: null,
    penaltyAmount: null,
    refundToken: null,
    trainsOrderType: null,
    selectedTickets: [],
    updateStatus: STATUS.NULL,
    refundStatus: STATUS.NULL,
    isRefundModalOpened: false,
    isRegistrationModalOpened: false,
    isAuthError: false,
    requestedRegistrationStatus: null,
};

export default createReducer<
    IOrdersState,
    ActionType<typeof actions> | TCheckOrderAuthorizationActionsType
>(initialState)
    .handleAction(actions.setSelectedTickets, (state, action) => {
        return {
            ...state,
            selectedTickets: action.payload,
        };
    })
    .handleAction(actions.setRefundModalOpenedState, (state, action) => {
        return {
            ...state,
            isRefundModalOpened: action.payload,
        };
    })
    .handleAction(actions.setRegistrationModalOpenedState, (state, action) => {
        return {
            ...state,
            isRegistrationModalOpened: action.payload,
        };
    })
    .handleAction(checkOrderAuthorizationActions.success, (state, action) => {
        const {orderType: authOrderType} = action.payload;

        const orderType = convertOrderType(authOrderType);

        return {
            ...state,
            trainsOrderType:
                orderType === ORDER_TYPE_VALUES.TRAINS ||
                orderType === ORDER_TYPE_VALUES.TRAINS_GENERIC
                    ? orderType
                    : null,
        };
    })
    .handleAction(checkOrderAuthorizationActions.failure, (state, action) => {
        const authOrderType = action.payload?.orderType ?? null;

        if (!authOrderType) {
            return state;
        }

        const orderType = convertOrderType(authOrderType);

        return {
            ...state,
            trainsOrderType:
                orderType === ORDER_TYPE_VALUES.TRAINS ||
                orderType === ORDER_TYPE_VALUES.TRAINS_GENERIC
                    ? orderType
                    : null,
        };
    })
    .handleAction(actions.setRefundAmountInfo, (state, action) => {
        const {refundAmount, refundToken, penaltyAmount} = action.payload;

        if (!state.trainsOrder) {
            return state;
        }

        return {
            ...state,
            trainsOrder: {
                ...state.trainsOrder,
                refundAmount,
                refundToken,
                penaltyAmount,
            },
            refundStatus: STATUS.NULL,
            refundToken,
            refundAmount,
            penaltyAmount,
        };
    })
    .handleAction(actions.updateOrder.request, (state, action) => {
        if (action.payload.isBackground) {
            return state;
        }

        return {
            ...state,
            isFetching: true,
            isFetched: false,
            isError: false,
        };
    })
    .handleAction(actions.updateOrder.success, (state, action) => {
        return {
            ...state,
            trainsOrder: action.payload,
            isFetching: false,
            isError: false,
            updateStatus: STATUS.NULL,
            refundStatus: STATUS.NULL,
        };
    })
    .handleAction(actions.updateOrder.failure, (state, action) => {
        return {
            ...state,
            error: action.payload,
            isFetching: false,
            isFetched: false,
            isError: true,
            updateStatus: STATUS.NULL,
            refundStatus: STATUS.NULL,
        };
    })
    .handleAction(
        [actions.orderRefundAmountRequest, actions.orderRefund.request],
        state => {
            return {
                ...state,
                refundStatus: STATUS.PENDING,
            };
        },
    )
    .handleAction(actions.orderRefund.failure, (state, action) => {
        return {
            ...state,
            error: action.payload,
            isError: true,
            refundStatus: STATUS.NULL,
        };
    })
    .handleAction(actions.orderRegistrationStatusUpdate.request, state => {
        return {
            ...state,
            updateStatus: STATUS.PENDING,
        };
    })
    .handleAction(
        actions.orderRegistrationStatusUpdate.success,
        (state, action) => {
            return {
                ...state,
                trainsOrder: action.payload,
                isFetching: false,
                isError: false,
                isRegistrationModalOpened: false,
                updateStatus: STATUS.RESOLVE,
            };
        },
    )
    .handleAction(
        actions.orderRegistrationStatusUpdate.failure,
        (state, action) => {
            return {
                ...state,
                error: action.payload,
                isError: true,
                updateStatus: STATUS.REJECT,
            };
        },
    )
    .handleAction(actions.clearError, state => {
        return {
            ...state,
            isFetching: false,
            isFetched: false,
            isError: false,
            updateStatus: STATUS.NULL,
            refundStatus: STATUS.NULL,
        };
    })
    .handleAction(actions.setRequestedRegistrationStatus, (state, action) => {
        return {
            ...state,
            requestedRegistrationStatus: action.payload,
        };
    });
