import {ActionType, createReducer} from 'typesafe-actions';

import {IPreparedPassenger} from 'reducers/account/passengers/api-types';

import prepareFetched from './utilities/prepareFetched';

import * as actions from './actions';

interface IPassengersStore {
    items: IPreparedPassenger[];
    lastInsertedId: string;
    fetching: boolean;
    fetched: boolean;
    error: boolean;
    authError: boolean;
}

const INITIAL_STATE: IPassengersStore = {
    items: [],
    lastInsertedId: '',
    fetching: false,
    fetched: false,
    error: false,
    authError: false,
};

export default createReducer<IPassengersStore, ActionType<typeof actions>>(
    INITIAL_STATE,
)
    .handleAction(actions.processPassenger, state => ({
        ...state,
        fetching: true,
        fetched: false,
        error: false,
        authError: false,
    }))
    .handleAction(actions.fetchPassengerSuccess, (state, action) => ({
        ...state,
        items: prepareFetched(action.payload),
        fetching: false,
        fetched: true,
        error: false,
        authError: false,
    }))
    .handleAction(actions.editPassengerSuccess, (state, action) => ({
        ...state,
        items: state.items.map(item => {
            if (item.id === action.payload.id) {
                return prepareFetched({
                    ...item,
                    ...action.payload,
                });
            }

            return item;
        }),
        fetched: true,
        fetching: false,
        error: false,
        authError: false,
    }))
    .handleAction(actions.addPassengerSuccess, (state, action) => ({
        items: [...state.items, prepareFetched(action.payload)],
        fetching: false,
        fetched: true,
        error: false,
        authError: false,
        lastInsertedId: action.payload.id,
    }))
    .handleAction(actions.removePassengerSuccess, (state, action) => ({
        ...state,
        items: state.items.filter(item => item.id !== action.payload.id),
        fetching: false,
        fetched: true,
        error: false,
        authError: false,
    }))
    .handleAction(actions.failPassenger, state => ({
        ...state,
        fetching: false,
        fetched: false,
        error: true,
        authError: false,
    }))
    .handleAction(actions.authErrorPassenger, state => ({
        ...state,
        fetching: false,
        fetched: false,
        error: false,
        authError: true,
    }))
    .handleAction(actions.redirectedToJustAdded, state => ({
        ...state,
        lastInsertedId: '',
    }));
