import {batch} from 'react-redux';

import TValidDataForRequestDynamic from 'types/avia/dynamic/TValidDataForRequestDynamic';

import {CustomThunkAction} from 'reducers/trains/customDispatch';
import * as actions from 'reducers/avia/aviaPriceIndex/dynamics/actions';
import * as weekPricesActions from 'reducers/avia/aviaPriceIndex/weekPrices/actions';
import {
    convertPriceIndexDataToDynamicsData,
    TDynamicsDaysInfo,
} from 'reducers/avia/aviaPriceIndex/utils/convertPriceIndexDataToDynamicsData';

import aviaDynamicsPricesCurrentRequestParamsSelector from 'selectors/avia/dynamicsPrices/aviaDynamicsPricesCurrentRequestParamsSelector';
import aviaWeekPricesCurrentRequestParamsSelector from 'selectors/avia/weekPrices/aviaWeekPricesCurrentRequestParamsSelector';
import {aviaWeekPricesPriceSelector} from 'selectors/avia/weekPrices/aviaWeekPricesSelectors';

import isCompatibleDynamicSearch from 'projects/avia/lib/dynamic/isCompatibleDynamicSearch';
import {unknownErrToString} from 'utilities/error';

import {aviaPriceIndexBrowserProvider} from 'serviceProvider/avia/aviaPriceIndexBrowserProvider';

/**
 * Реализует запрос за данными динамики для state.avia.aviaPriceIndex.dynamicPrices, но
 * так же может обновить и state.avia.aviaPriceIndex.weekPrices, если данные совместимы по
 * исходным данным для запроса.
 */
export function dynamicsRequest(
    dataForRequest: TValidDataForRequestDynamic,
): CustomThunkAction<void> {
    return async (dispatch, getState): Promise<void> => {
        const {searchForm, filters, interval} = dataForRequest;

        dispatch(actions.dynamicsSearchRequest(dataForRequest));

        try {
            const result = await aviaPriceIndexBrowserProvider.dynamics(
                interval,
                searchForm,
                filters,
            );

            const actualState = getState();
            const currentDynamicsPricesRequestParams =
                aviaDynamicsPricesCurrentRequestParamsSelector(actualState);
            const currentWeekPricesRequestParams =
                aviaWeekPricesCurrentRequestParamsSelector(actualState);

            const dynamicPricesCompatible =
                !currentDynamicsPricesRequestParams ||
                isCompatibleDynamicSearch(
                    dataForRequest,
                    currentDynamicsPricesRequestParams,
                );
            const weekPricesCompatible =
                !currentWeekPricesRequestParams ||
                isCompatibleDynamicSearch(
                    currentWeekPricesRequestParams,
                    dataForRequest,
                );

            const data = convertPriceIndexDataToDynamicsData(result);

            batch(() => {
                const dataForAction = {
                    data,
                    ...dataForRequest,
                };

                if (dynamicPricesCompatible) {
                    dispatch(actions.dynamicsSearchSuccess(dataForAction));
                }

                if (weekPricesCompatible) {
                    // Дополняем данные о ценах из weekPrice, полученные через поиск или inline-поиск
                    const weekPrices = aviaWeekPricesPriceSelector(actualState);

                    if (weekPrices) {
                        const searchPrices = Object.entries(
                            weekPrices,
                        ).reduce<TDynamicsDaysInfo>(
                            (resPrices, [key, price]) => {
                                if (price.fromSearch) {
                                    resPrices[key] = price;
                                }

                                return resPrices;
                            },
                            {},
                        );

                        dispatch(
                            actions.updatePrices({
                                ...dataForAction,
                                data: searchPrices,
                            }),
                        );
                    }

                    // Обновляем цены weekPrice из dynamicPrice
                    dispatch(weekPricesActions.updatePrices(dataForAction));
                }
            });
        } catch (e) {
            dispatch(
                actions.dynamicsSearchFailure({
                    error: unknownErrToString(e),
                    ...dataForRequest,
                }),
            );
            console.log(e);
        }
    };
}
