import {batch} from 'react-redux';

import TValidSearchForm from 'types/avia/dynamic/TValidSearchForm';
import TValidDataForRequestDynamic from 'types/avia/dynamic/TValidDataForRequestDynamic';

import {CustomThunkAction} from 'reducers/trains/customDispatch';
import * as actions from 'reducers/avia/aviaPriceIndex/dynamics/actions';
import * as weekPricesActions from 'reducers/avia/aviaPriceIndex/weekPrices/actions';
import * as tableDynamicActions from 'reducers/avia/aviaPriceIndex/tableDynamic/actions';
import {AviaMinPriceFinder} from 'reducers/avia/aviaPriceIndex/utils/AviaMinPriceFinder';
import {convertInlineSearchDataToDynamicsDay} from 'reducers/avia/aviaPriceIndex/utils/convertInlineSearchDataToDynamicsDay';
import {IAviaSearchResultsFilters} from 'reducers/avia/search/results/filters/reducer';

import {currenciesConverterSelector} from 'selectors/common/currenciesSelector';
import {aviaDynamicsPricesRequestParamsSelector} from 'selectors/avia/dynamicsPrices/aviaDynamicsPricesRequestParamsSelector';
import aviaDynamicsPricesCurrentRequestParamsSelector from 'selectors/avia/dynamicsPrices/aviaDynamicsPricesCurrentRequestParamsSelector';
import aviaWeekPricesCurrentRequestParamsSelector from 'selectors/avia/weekPrices/aviaWeekPricesCurrentRequestParamsSelector';
import tableDynamicCurrentRequestParamsSelector from 'selectors/avia/tableDynamic/tableDynamicCurrentRequestParamsSelector';

import IPrice from 'utilities/currency/PriceInterface';
import isCompatibleDynamicSearch from 'projects/avia/lib/dynamic/isCompatibleDynamicSearch';
import isDateRobot from 'utilities/dateUtils/isDateRobot';
import isCompatibleDynamicSearchWithTableDynamic from 'projects/avia/lib/dynamic/isCompatibleDynamicSearchWithTableDynamic';
import {unknownErrToString} from 'utilities/error';

export type TInlineSearchRequestAction = (
    forwardDate: string,
    backwardDate: Nullable<string>,

    // Кастомная поисковая форма. Иначе возьмется актуальная для динамики цен
    searchForm?: TValidSearchForm,
    // Кастомные фильтры. Иначе возьмутся актуальные для динамики цен
    filters?: IAviaSearchResultsFilters,
) => CustomThunkAction<void>;

export const inlineSearchRequest: TInlineSearchRequestAction = (
    forwardDate,
    backwardDate,

    searchForm,
    filters,
) => {
    return async (dispatch, getState): Promise<void> => {
        if (!isDateRobot(forwardDate)) {
            return;
        }

        const state = getState();
        const dynamicsRequestParams =
            aviaDynamicsPricesRequestParamsSelector(state);
        const currenciesConverter = currenciesConverterSelector(state);

        const searchFormForUse =
            searchForm || dynamicsRequestParams?.searchForm;
        const filtersForUse = filters || dynamicsRequestParams?.filters;

        if (!searchFormForUse || !filtersForUse) {
            return;
        }

        const searchFormForRequest = {
            ...searchFormForUse,
            when: forwardDate,
            return_date: backwardDate || '',
        };
        const inlineRequestParams: TValidDataForRequestDynamic = {
            filters: filtersForUse,
            searchForm: searchFormForRequest,
            interval: {startDate: forwardDate, endDate: forwardDate},
        };

        const isActualForDynamic = (): boolean => {
            const currentRequestParamsDynamic =
                aviaDynamicsPricesCurrentRequestParamsSelector(getState());

            return Boolean(
                currentRequestParamsDynamic &&
                    isCompatibleDynamicSearch(
                        inlineRequestParams,
                        currentRequestParamsDynamic,
                    ),
            );
        };

        const isActualForWeekPrice = (): boolean => {
            const currentRequestParamsWeekPrices =
                aviaWeekPricesCurrentRequestParamsSelector(getState());

            return Boolean(
                currentRequestParamsWeekPrices &&
                    isCompatibleDynamicSearch(
                        inlineRequestParams,
                        currentRequestParamsWeekPrices,
                    ),
            );
        };

        const isActualForTableDynamic = (): boolean => {
            const currentRequestParamsDynamicTable =
                tableDynamicCurrentRequestParamsSelector(getState());

            return Boolean(
                currentRequestParamsDynamicTable &&
                    isCompatibleDynamicSearchWithTableDynamic(
                        inlineRequestParams,
                        currentRequestParamsDynamicTable,
                    ),
            );
        };

        const maybeApplyForDynamic = isActualForDynamic();
        const maybeApplyForWeekPrice = isActualForWeekPrice();
        const maybeApplyForDynamicTable = isActualForTableDynamic();

        if (
            !maybeApplyForDynamic &&
            !maybeApplyForWeekPrice &&
            !maybeApplyForDynamicTable
        ) {
            return;
        }

        batch(() => {
            if (maybeApplyForDynamic) {
                dispatch(actions.inlineSearchRequest(inlineRequestParams));
            }

            if (maybeApplyForWeekPrice) {
                dispatch(
                    weekPricesActions.inlineSearchRequest(inlineRequestParams),
                );
            }

            if (maybeApplyForDynamicTable) {
                dispatch(
                    tableDynamicActions.inlineSearchRequest(
                        inlineRequestParams,
                    ),
                );
            }
        });

        const onPriceFounded = (
            price: IPrice | null,
            progress: number,
            finder: AviaMinPriceFinder,
            searchInProgress: boolean,
        ): void => {
            const actualForDynamic = isActualForDynamic();
            const actualForWeekPrice = isActualForWeekPrice();
            const actualForTableDynamic = isActualForTableDynamic();

            if (
                !actualForDynamic &&
                !actualForWeekPrice &&
                !actualForTableDynamic
            ) {
                return finder.stop();
            }

            const data = convertInlineSearchDataToDynamicsDay(
                price,
                progress,
                searchInProgress,
            );

            const dataForAction = {
                data,
                ...inlineRequestParams,
            };

            batch(() => {
                if (actualForDynamic) {
                    if (searchInProgress) {
                        dispatch(
                            actions.inlineSearchPriceFounded(dataForAction),
                        );
                    } else {
                        dispatch(actions.inlineSearchSuccess(dataForAction));
                    }
                }

                if (actualForWeekPrice) {
                    if (searchInProgress) {
                        dispatch(
                            weekPricesActions.inlineSearchPriceFounded(
                                dataForAction,
                            ),
                        );
                    } else {
                        dispatch(
                            weekPricesActions.inlineSearchSuccess(
                                dataForAction,
                            ),
                        );
                    }
                }

                if (actualForTableDynamic) {
                    if (searchInProgress) {
                        dispatch(
                            tableDynamicActions.inlineSearchPriceFounded(
                                dataForAction,
                            ),
                        );
                    } else {
                        dispatch(
                            tableDynamicActions.inlineSearchSuccess(
                                dataForAction,
                            ),
                        );
                    }
                }
            });
        };

        const finder = new AviaMinPriceFinder({
            searchForm: inlineRequestParams.searchForm,
            filters: inlineRequestParams.filters,
            onPriceFounded: (price, progress, finderObj): void =>
                onPriceFounded(price, progress, finderObj, true),
            currenciesConverter,
        });

        try {
            const {price, progress} = await finder.search();

            onPriceFounded(price, progress, finder, false);
        } catch (e) {
            const dataForAction = {
                error: unknownErrToString(e),
                ...inlineRequestParams,
            };

            batch(() => {
                dispatch(actions.inlineSearchFailure(dataForAction));
                dispatch(weekPricesActions.inlineSearchFailure(dataForAction));
                dispatch(
                    tableDynamicActions.inlineSearchFailure(dataForAction),
                );
            });
        }
    };
};
