import {ActionType, createReducer} from 'typesafe-actions';

import EAsyncStatus from 'types/common/EAsyncStatus';
import EAviaDynamicTypeView from 'types/avia/priceIndex/EAviaDynamicTypeView';
import TValidSearchForm from 'types/avia/dynamic/TValidSearchForm';

import {IAviaContext} from 'reducers/avia/data-types';
import {IAviaSearchResultsFilters} from 'reducers/avia/search/results/filters/reducer';
import dynamicSearchRequestHandler from 'reducers/avia/aviaPriceIndex/utils/dynamicSearchRequestHandler';
import dynamicSearchSuccessHandler from 'reducers/avia/aviaPriceIndex/utils/dynamicSearchSuccessHandler';
import dynamicSearchFailureHandler from 'reducers/avia/aviaPriceIndex/utils/dynamicSearchFailureHandler';
import dynamicUpdatePrices from 'reducers/avia/aviaPriceIndex/utils/dynamicUpdatePrices';
import dynamicInlineSearchRequestHandler from 'reducers/avia/aviaPriceIndex/utils/dynamicInlineSearchRequestHandler';
import dynamicInlineSearchPriceFoundedHandler from 'reducers/avia/aviaPriceIndex/utils/dynamicInlineSearchPriceFoundedHandler';
import dynamicInlineSearchFailureHandler from 'reducers/avia/aviaPriceIndex/utils/dynamicInlineSearchFailureHandler';

import {TDynamicsDaysInfo} from '../utils/convertPriceIndexDataToDynamicsData';
import {TInterval} from 'projects/avia/components/Dynamics/utilities/getAviaDynamicsCalendarDates';

import * as actions from './actions';
import aviaContextReducer, {
    AVIA_SEARCH_FORM_INITIAL_STATE,
} from '../../context/reducer';
import {
    resetAviaContext,
    setAviaContext,
    TAviaContextActions,
} from '../../context/actions';

export interface IAviaDynamicsState {
    // Статус запроса к динамике цен
    status: EAsyncStatus | null;
    // Собственная поисковая форма. В целом нужна только для того, чтобы менять длительность поездки в шторке динамики
    searchForm: IAviaContext;
    // Признак того, что для конкретной выдачи шторка динамики открывалась хотя бы раз
    wasOpened: boolean;
    // Цены
    prices: TDynamicsDaysInfo | null;
    // Ошибка при запросе
    error: string | null;
    // Данные поисковой формы, для которой произведен/производится запрос
    forSearchForm: TValidSearchForm | null;
    // Данные фильтров, для которых произведен/производится запрос
    forFilters: IAviaSearchResultsFilters | null;
    // Интервал дней, для которого произведен/производится запрос
    forInterval: TInterval | null;
    // Тип отображения (график или таблица)
    typeView: EAviaDynamicTypeView;
}

export default createReducer<
    IAviaDynamicsState,
    ActionType<typeof actions> | TAviaContextActions
>({
    status: null,
    searchForm: AVIA_SEARCH_FORM_INITIAL_STATE,
    wasOpened: false,
    prices: null,
    error: null,
    forSearchForm: null,
    forFilters: null,
    forInterval: null,
    typeView: EAviaDynamicTypeView.TABLE,
})
    .handleAction(actions.dynamicsSearchRequest, dynamicSearchRequestHandler)
    .handleAction(actions.dynamicsSearchSuccess, dynamicSearchSuccessHandler)
    .handleAction(actions.dynamicsSearchFailure, dynamicSearchFailureHandler)
    .handleAction(actions.updatePrices, dynamicUpdatePrices)
    .handleAction(
        actions.inlineSearchRequest,
        dynamicInlineSearchRequestHandler,
    )
    .handleAction(
        [actions.inlineSearchSuccess, actions.inlineSearchPriceFounded],
        dynamicInlineSearchPriceFoundedHandler,
    )
    .handleAction(
        actions.inlineSearchFailure,
        dynamicInlineSearchFailureHandler,
    )
    .handleAction([setAviaContext, resetAviaContext], (state, action) => ({
        ...state,
        searchForm: aviaContextReducer(state.searchForm, action),
        prices: null,
        status: null,
        wasOpened: false,
        forSearchForm: null,
        forFilters: null,
        forInterval: null,
    }))
    .handleAction(actions.changeBackwardDate, (state, {payload}) => ({
        ...state,
        searchForm: {
            ...state.searchForm,
            return_date: payload,
        },
        prices: null,
        status: null,
        forSearchForm: null,
        forFilters: null,
        forInterval: null,
    }))
    .handleAction(actions.setWasOpened, (state, {payload}) => ({
        ...state,
        wasOpened: payload,
    }))
    .handleAction(actions.setTypeView, (state, {payload}) => ({
        ...state,
        typeView: payload,
    }));
