/* eslint-disable camelcase */
import {ActionType, createReducer} from 'typesafe-actions';

import EAsyncStatus from 'types/common/EAsyncStatus';
import {EDynamicsDayStatus} from 'projects/avia/components/Dynamics/types/IDynamicsDay';
import TValidRoundTripSearchForm from 'types/avia/dynamic/TValidRoundTripSearchForm';

import {IAviaSearchResultsFilters} from 'reducers/avia/search/results/filters/reducer';

import {TDynamicTableDaysInfo} from '../utils/convertPriceIndexDataToDynamicsData';
import getKeyForTablePrice from 'projects/avia/lib/dynamic/getKeyForTablePrice';
import isDateRobot from 'utilities/dateUtils/isDateRobot';

import * as actions from './actions';

export interface IAviaTableDynamicsState {
    // Статус запроса к динамике цен
    status: EAsyncStatus | null;
    // Цены
    prices: TDynamicTableDaysInfo | null;
    // Ошибка при запросе
    error: string | null;
    // Данные поисковой формы, для которой произведен/производится запрос
    forSearchForm: TValidRoundTripSearchForm | null;
    // Данные фильтров, для которых произведен/производится запрос
    forFilters: IAviaSearchResultsFilters | null;
}

export default createReducer<
    IAviaTableDynamicsState,
    ActionType<typeof actions>
>({
    status: null,
    prices: null,
    error: null,
    forSearchForm: null,
    forFilters: null,
})
    .handleAction(actions.dynamicsSearchRequest, (state, {payload}) => ({
        ...state,
        status: EAsyncStatus.LOADING,
        prices: null,
        error: null,
        forSearchForm: payload.searchForm,
        forFilters: payload.filters,
    }))
    .handleAction(actions.dynamicsSearchSuccess, (state, {payload}) => ({
        ...state,
        status: EAsyncStatus.SUCCESS,
        error: null,
        forSearchForm: payload.searchForm,
        forFilters: payload.filters,
        prices: payload.data,
    }))
    .handleAction(actions.dynamicsSearchFailure, (state, {payload}) => ({
        ...state,
        status: EAsyncStatus.ERROR,
        error: payload.error,
        forSearchForm: payload.searchForm,
        forFilters: payload.filters,
        prices: null,
    }))
    .handleAction(actions.updatePrices, (state, {payload}) => ({
        ...state,
        prices: {
            ...state.prices,
            ...payload.data,
        },
    }))
    .handleAction(actions.inlineSearchRequest, (state, {payload}) => {
        const {
            searchForm: {when, return_date},
        } = payload;

        if (!isDateRobot(return_date)) {
            return state;
        }

        const key = getKeyForTablePrice(when, return_date);

        return {
            ...state,
            prices: {
                ...state.prices,
                [key]: {
                    price: null,
                    ...state.prices?.[key],
                    status: EDynamicsDayStatus.SEARCHING,
                    progress: 0,
                    fromSearch: true,
                },
            },
        };
    })
    .handleAction(
        [actions.inlineSearchSuccess, actions.inlineSearchPriceFounded],
        (state, {payload}) => {
            const {
                searchForm: {when, return_date},
                data,
            } = payload;

            if (!isDateRobot(return_date)) {
                return state;
            }

            const key = getKeyForTablePrice(when, return_date);

            return {
                ...state,
                prices: {
                    ...state.prices,
                    [key]: {
                        ...data,
                        fromSearch: true,
                    },
                },
            };
        },
    )
    .handleAction(actions.inlineSearchFailure, (state, {payload}) => {
        const {
            searchForm: {when, return_date},
        } = payload;

        if (!isDateRobot(return_date)) {
            return state;
        }

        const key = getKeyForTablePrice(when, return_date);

        return {
            ...state,
            prices: {
                ...state.prices,
                [key]: {
                    price: null,
                    progress: 0,
                    ...state.prices?.[key],
                    status: EDynamicsDayStatus.WAS_ERROR,
                    fromSearch: false,
                },
            },
        };
    });
