import moment from 'moment';

import TValidDataForRequestTableDynamic from 'types/avia/dynamic/TValidDataForRequestTableDynamic';
import TValidDataForRequestDynamic from 'types/avia/dynamic/TValidDataForRequestDynamic';
import {IDataForRequestDynamic} from 'types/avia/dynamic/IDataForRequestDynamic';

import {CustomThunkAction} from 'reducers/trains/customDispatch';
import {
    convertPriceIndexDataToDynamicTableDaysInfo,
    TDynamicsDaysInfo,
    TDynamicTableDaysInfo,
} from 'reducers/avia/aviaPriceIndex/utils/convertPriceIndexDataToDynamicsData';
import {
    dynamicsSearchFailure,
    dynamicsSearchRequest,
    dynamicsSearchSuccess,
} from 'reducers/avia/aviaPriceIndex/tableDynamic/actions';

import tableDynamicCurrentRequestParamsSelector from 'selectors/avia/tableDynamic/tableDynamicCurrentRequestParamsSelector';
import aviaWeekPricesCurrentRequestParamsSelector from 'selectors/avia/weekPrices/aviaWeekPricesCurrentRequestParamsSelector';
import {aviaWeekPricesPriceSelector} from 'selectors/avia/weekPrices/aviaWeekPricesSelectors';
import aviaDynamicsPricesCurrentRequestParamsSelector from 'selectors/avia/dynamicsPrices/aviaDynamicsPricesCurrentRequestParamsSelector';
import {aviaDynamicsPricesPriceSelector} from 'selectors/avia/dynamicsPrices/aviaDynamicsPricesSelectors';

import getIntervalsForRequestTableDynamic, {
    IGetIntervalsForRequestTableDynamicCoords,
} from 'projects/avia/lib/dynamic/getIntervalsForRequestTableDynamic';
import isValidDataForRequestDynamic from 'projects/avia/lib/dynamic/isValidDataForRequestDynamic';
import isCompatibleTableDynamicSearch from 'projects/avia/lib/dynamic/isCompatibleTableDynamicSearch';
import isCompatibleDynamicSearchWithTableDynamic from 'projects/avia/lib/dynamic/isCompatibleDynamicSearchWithTableDynamic';
import getKeyForTablePrice from 'projects/avia/lib/dynamic/getKeyForTablePrice';
import isDateRobot from 'utilities/dateUtils/isDateRobot';
import getDatesDelta from 'projects/avia/lib/dynamic/getDatesDelta';
import getDateRobotFromMoment from 'utilities/dateUtils/getDateRobotFromMoment';
import {unknownErrToString} from 'utilities/error';

import {aviaPriceIndexBrowserProvider} from 'serviceProvider/avia/aviaPriceIndexBrowserProvider';

// Допустимый процент сфэйлившихся запросов к динамике цен
const ACCEPTABLE_ERROR_PERCENT = 20;

interface IDynamicQueryResult {
    error: string | null;
    prices: TDynamicTableDaysInfo | null;
}

export interface ITableDynamicRequest
    extends IGetIntervalsForRequestTableDynamicCoords {
    requestParams: TValidDataForRequestTableDynamic;
}

export default function tableDynamicRequest({
    requestParams,
    leftRange,

    rightRange,
    topRange,
    bottomRange,
}: ITableDynamicRequest): CustomThunkAction<void> {
    return async (dispatch, getState): Promise<void> => {
        try {
            const {searchForm, filters} = requestParams;
            const {when: whenSearchForm, return_date: returnDateSearchForm} =
                searchForm;

            dispatch(dynamicsSearchRequest(requestParams));

            const tableIntervals = getIntervalsForRequestTableDynamic({
                forwardDate: whenSearchForm,
                backwardDate: returnDateSearchForm,
                leftRange,
                rightRange,
                topRange,
                bottomRange,
            });

            const dataForRequests: TValidDataForRequestDynamic[] =
                tableIntervals.reduce<TValidDataForRequestDynamic[]>(
                    (validParams, {when, returnDate, startDate, endDate}) => {
                        const reqParams: IDataForRequestDynamic = {
                            searchForm: {
                                ...searchForm,
                                when,
                                return_date: returnDate,
                            },
                            filters,
                            interval: {
                                startDate,
                                endDate,
                            },
                        };

                        if (isValidDataForRequestDynamic(reqParams)) {
                            validParams.push(reqParams);
                        }

                        return validParams;
                    },
                    [],
                );

            const requests: Promise<IDynamicQueryResult>[] =
                dataForRequests.map(reqParams => {
                    return aviaPriceIndexBrowserProvider
                        .dynamics(
                            reqParams.interval,
                            reqParams.searchForm,
                            reqParams.filters,
                        )
                        .then(result => ({
                            error: null,
                            prices: convertPriceIndexDataToDynamicTableDaysInfo(
                                result,
                                reqParams.searchForm,
                            ),
                        }))
                        .catch(error => ({
                            error: error.message || String(error),
                            prices: null,
                        }));
                });

            const results: IDynamicQueryResult[] = await Promise.all(requests);

            // Проверяем, что запрос еще актуален
            const currentRequestParams =
                tableDynamicCurrentRequestParamsSelector(getState());

            if (
                !currentRequestParams ||
                !isCompatibleTableDynamicSearch(
                    requestParams,
                    currentRequestParams,
                )
            ) {
                return;
            }

            const countFailed = results.reduce((failed, {error}) => {
                if (error) {
                    // eslint-disable-next-line no-param-reassign
                    failed++;
                }

                return failed;
            }, 0);

            const percentFailed = (countFailed / results.length) * 100;

            if (percentFailed > ACCEPTABLE_ERROR_PERCENT) {
                throw new Error(
                    `Most of the requests failed. Percent: ${percentFailed}`,
                );
            }

            // Формируем итоговый объект с данными
            const finalPrices: TDynamicTableDaysInfo = Object.assign(
                {},
                ...Object.values(results).map(result => result.prices),
            );

            // Дополняем данные о ценах из weekPrice
            const currentRequestParamsWeekPrices =
                aviaWeekPricesCurrentRequestParamsSelector(getState());
            const weekPrices = aviaWeekPricesPriceSelector(getState());

            if (weekPrices && currentRequestParamsWeekPrices) {
                const pricesFromWeekPrices =
                    getTablePricesFromWeekPricesOrDynamicsPrices(
                        requestParams,
                        currentRequestParamsWeekPrices,
                        weekPrices,
                    );

                if (pricesFromWeekPrices) {
                    Object.assign(finalPrices, pricesFromWeekPrices);
                }
            }

            // Дополняем данные о ценах из dynamicPrice
            const currentRequestParamsDynamicsPrices =
                aviaDynamicsPricesCurrentRequestParamsSelector(getState());
            const dynamicsPrices = aviaDynamicsPricesPriceSelector(getState());

            if (dynamicsPrices && currentRequestParamsDynamicsPrices) {
                const pricesFromDynamicsPrices =
                    getTablePricesFromWeekPricesOrDynamicsPrices(
                        requestParams,
                        currentRequestParamsDynamicsPrices,
                        dynamicsPrices,
                    );

                if (pricesFromDynamicsPrices) {
                    Object.assign(finalPrices, pricesFromDynamicsPrices);
                }
            }

            dispatch(
                dynamicsSearchSuccess({data: finalPrices, ...requestParams}),
            );
        } catch (e) {
            dispatch(
                dynamicsSearchFailure({
                    error: unknownErrToString(e),
                    ...requestParams,
                }),
            );
            console.log(e);
        }
    };
}

function getTablePricesFromWeekPricesOrDynamicsPrices(
    currentTableRequestParams: TValidDataForRequestTableDynamic,
    currentRequestParams: TValidDataForRequestDynamic,
    prices: TDynamicsDaysInfo,
): TDynamicTableDaysInfo | null {
    const resultPrices: TDynamicsDaysInfo = {};

    if (
        !isCompatibleDynamicSearchWithTableDynamic(
            currentRequestParams,
            currentTableRequestParams,
        )
    ) {
        return null;
    }

    // eslint-disable-next-line camelcase
    const {searchForm} = currentRequestParams;
    const dayDelta = getDatesDelta(searchForm);

    if (!dayDelta) {
        return null;
    }

    Object.entries(prices).forEach(([when, dynamicsDay]) => {
        if (!dynamicsDay.fromSearch || !isDateRobot(when)) {
            return;
        }

        const returnDateMoment = moment(when).add(dayDelta, 'days');
        const key = getKeyForTablePrice(
            when,
            getDateRobotFromMoment(returnDateMoment),
        );

        resultPrices[key] = dynamicsDay;
    });

    return resultPrices;
}
