/**
 * При переносе этого модуля использовался старый подход к выполненю запросов, там
 * можно было обрывать запросы. Клиенты поверх BrowserHttpClient сейчас не умеют такого,
 * но на будущее оставил логику с этим (кроме проброса токена до самого аксиоса).
 */
import axios, {CancelTokenSource} from 'axios';

import {IAviaParams} from 'server/services/AviaSearchService/types/IAviaParams';
import {IAviaTDInitAnswer} from 'server/api/AviaTicketDaemonApi/types/IAviaTDInitAnswer';
import {IAviaTDAnswer} from 'server/api/AviaTicketDaemonApi/types/IAviaTDAnswer';

import {
    getInitialState,
    IAviaSearchResultsFilters,
} from 'reducers/avia/search/results/filters/reducer';
import {normalizeTDAnswer} from 'reducers/avia/utils/ticketDaemon/normalizeTDAnswer';

import {processNormalizedVariants} from 'selectors/avia/search/getDenormalizedVariants';
import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';

import IPrice from 'utilities/currency/PriceInterface';
import {PriceConverter} from 'utilities/currency/priceConverter';
import {PriceComparator} from 'utilities/currency/compare';
import {aviaFilterBy} from 'projects/avia/lib/search/filters/filterVariants';

import aviaBrowserProvider from 'serviceProvider/avia/aviaBrowserProvider';

export interface ILastData {
    price: Nullable<IPrice>;
    /** Целое число от 0 до 100 */
    progress: number;
}

export type TOnPriceFounded = (
    price: IPrice,
    /** Целое число от 0 до 100 */
    progress: number,
    priceFinder: AviaMinPriceFinder,
) => void;

interface IAviaMinPriceFinderOptions {
    searchForm: IAviaParams;
    filters: Partial<IAviaSearchResultsFilters>;
    onPriceFounded: TOnPriceFounded;
    currenciesConverter: PriceConverter;
}

export class AviaMinPriceFinder {
    private static readonly TRIES = 3;
    private static readonly REQUEST_WITHOUT_VARIANTS_LIMIT = 20000;
    private static readonly TOTAL_SEARCH_TIME_LIMIT = 60000;
    private static readonly WAIT_TIMEOUT = 1000;

    private qid: string | undefined = undefined;
    private cont: Nullable<number> = null;

    private lastData: ILastData = {
        price: null,
        progress: 0,
    };

    // Это ретраи для случаев, когда ответ тд нас не устроил
    private searchTries = AviaMinPriceFinder.TRIES;

    // Это ретраи для отдельного запроса на случай падения тд или сетевых ошибок
    private triesPerRequest = AviaMinPriceFinder.TRIES;

    private isStopped = false;
    private cancelToken: CancelTokenSource | undefined = undefined;

    private lastAnswerTime: number = Date.now();
    private fetcherInitTime: number = Date.now();

    private searchForm: IAviaParams;
    private filters: Partial<IAviaSearchResultsFilters>;
    private onPriceFounded: TOnPriceFounded;
    private priceComparator: PriceComparator;

    constructor({
        searchForm,
        filters,
        onPriceFounded,
        currenciesConverter,
    }: IAviaMinPriceFinderOptions) {
        this.searchForm = searchForm;
        this.filters = filters;
        this.onPriceFounded = onPriceFounded;
        this.priceComparator = new PriceComparator(currenciesConverter);
    }

    search = () => {
        return this.initSearch(this.searchForm);
    };

    stop = () => {
        this.isStopped = true;

        if (this.cancelToken) {
            this.cancelToken.cancel();
        }
    };

    private initSearch = (searchForm: IAviaParams) => {
        this.cancelToken = axios.CancelToken.source();

        return this.getQid(searchForm).then(this.initFetcher);
    };

    private initFetcher = (initData: IAviaTDInitAnswer) => {
        this.qid = initData.id;

        return this.fetchData();
    };

    private fetchData = async (): Promise<ILastData> => {
        if (this.isStopped || !this.qid) {
            return this.lastData;
        }

        const cont = this.cont || 0;
        const qid = this.qid;

        this.cancelToken = axios.CancelToken.source();

        return this.getData(qid, cont)
            .then(this.parseData)
            .catch(this.retryFetch);
    };

    private parseData = (data: IAviaTDAnswer) => {
        if (!data) {
            return this.retryFetch();
        }

        const progress = Math.floor(
            (data.progress.current / data.progress.all) * 100,
        );
        const minPrice = this.findMinPriceFromTD(data);

        if (minPrice) {
            // Данные получили - сбрасываем кол-во ретраев к дефолту
            this.reinitSearchTries();
        }

        if (this.cont !== data.cont) {
            this.cont = data.cont;

            if (minPrice) {
                this.lastAnswerTime = Date.now();
            }
        }

        // кажется, это нужно делать в других местах(как минимум до поиска мин.цены
        // или после того как мин цена записана в lastData)
        // и другим способом (абортить запросы по таймеру)
        if (this.isTooLong()) {
            return this.onFinish();
        }

        if (!this.lastData || !this.lastData.price) {
            this.lastData = {
                price: minPrice,
                progress,
            };
        } else if (minPrice) {
            this.lastData = {
                price: this.priceComparator.min(minPrice, this.lastData.price),
                progress,
            };
        }

        if (this.lastData.price) {
            this.onPriceFounded(
                this.lastData.price,
                this.lastData.progress,
                this,
            );
        }

        if (progress === 100 || data.cont === null) {
            return this.onFinish();
        }

        return this.waitForTimeout(AviaMinPriceFinder.WAIT_TIMEOUT).then(
            this.fetchData,
        );
    };

    private getFilteredVarinats = (
        variants: IResultAviaVariant[],
    ): Record<string, IResultAviaVariant> => {
        const filters = {
            ...getInitialState(),
            ...this.filters,
        };

        return Object.assign(
            {},
            aviaFilterBy.airports(filters.airports, variants),
            aviaFilterBy.baggage(filters.baggage, variants),
            aviaFilterBy.company(filters.company, variants),
            aviaFilterBy.price(filters.price, variants, this.priceComparator),
            aviaFilterBy.time(filters.time, variants),
            aviaFilterBy.transfer(filters.transfer, variants),
            aviaFilterBy.partner(filters.partners, variants),
        );
    };

    private findMinPriceFromTD = (data: IAviaTDAnswer) => {
        const normalisedAnswer = normalizeTDAnswer(data);
        const normVariants = processNormalizedVariants(
            normalisedAnswer.variants,
            normalisedAnswer.reference,
        );
        const filteredVariants = this.getFilteredVarinats(normVariants);

        const minPrice = normVariants.reduce((price, variant) => {
            if (filteredVariants[variant.tag]) {
                return price;
            }

            if (!price) {
                return variant.price.tariff;
            }

            return this.priceComparator.min(price, variant.price.tariff);
        }, null as IPrice | null);

        return minPrice;
    };

    private waitForTimeout = (timeout: number) =>
        new Promise(resolve => setTimeout(resolve, timeout));

    private isTooLong = () => {
        if (
            Date.now() - this.lastAnswerTime >
            AviaMinPriceFinder.REQUEST_WITHOUT_VARIANTS_LIMIT
        ) {
            return true;
        }

        if (
            Date.now() - this.fetcherInitTime >
            AviaMinPriceFinder.TOTAL_SEARCH_TIME_LIMIT
        ) {
            return true;
        }

        return false;
    };

    private retryFetch = () => {
        return this.waitForTimeout(AviaMinPriceFinder.WAIT_TIMEOUT).then(
            this.tryRetry,
        );
    };

    private tryRetry = () => {
        if (--this.searchTries > 0) {
            return this.fetchData();
        }

        throw new Error('Retry count exceeded');
    };

    private onFinish = () => {
        // Спорный момент. Возможно стоит явно сообщать что поиск прекращен
        return {
            ...this.lastData,
            progress: 100,
        };
    };

    private reinitSearchTries = () => {
        this.searchTries = AviaMinPriceFinder.TRIES;
    };

    private getQid = (searchForm: IAviaParams) => {
        return aviaBrowserProvider.initSearch(searchForm);
    };

    private getData = (qid: string, cont: number) => {
        return this.withRetry(() =>
            aviaBrowserProvider.getSearchResults(qid, cont, 0, {
                timeout: 55000,
            }),
        );
    };

    private withRetry<T>(cb: () => Promise<T>) {
        return cb().then(
            data => {
                this.triesPerRequest = AviaMinPriceFinder.TRIES;

                return data;
            },
            _ => this.handleFetchRetry().then(cb),
        );
    }

    private handleFetchRetry = () => {
        if (--this.triesPerRequest > 0) {
            return this.waitForTimeout(AviaMinPriceFinder.WAIT_TIMEOUT);
        }

        throw new Error('Retry count exceeded');
    };
}
