import moment from 'moment-timezone';

import {IAviaPriceIndexPrice} from 'server/api/AviaPriceIndexApi/types/IAviaPriceIndexPrice';
import {TAviaPriceIndexDynamicsData} from 'server/api/AviaPriceIndexApi/types/TAviaPriceIndexDynamicsData';
import {EDynamicsDayStatus} from 'projects/avia/components/Dynamics/types/IDynamicsDay';
import TValidSearchForm from 'types/avia/dynamic/TValidSearchForm';

import getDatesDelta from 'projects/avia/lib/dynamic/getDatesDelta';
import getKeyForTablePrice from 'projects/avia/lib/dynamic/getKeyForTablePrice';
import getDateRobotFromMoment from 'utilities/dateUtils/getDateRobotFromMoment';
import isDateRobot from 'utilities/dateUtils/isDateRobot';

export interface IDynamicsDay {
    status: EDynamicsDayStatus;
    price: Nullable<IAviaPriceIndexPrice>;
    // Прогресс inline-поиска. Целое число от 0 до 100
    progress: number;
    // Признак, что результат получен с помощью поиска на странице или inline-поиска. Считаем такой результат более приоритетным, чем данные из priceIndex
    fromSearch: boolean;
}

// ключ - дата YYYY-MM-DD (TDateRobot)
export type TDynamicsDaysInfo = Record<string, IDynamicsDay>;

// ключ - строка с датами туда-обратно 'YYYY-MM-DD:YYYY-MM-DD' (TDateRobot:TDateRobot)
export type TDynamicTableDaysInfo = Record<string, IDynamicsDay>;

/**
 * Трансформирует DTO ответа прайс индекса под удобный для дней динамики объект.
 * Из DTO достается только цена(если есть), выбирается статус. Эти статусы используются для UI.
 * При этом здесь только данные о ценах, а массив дней формируется отдельно, т.к. нужно еще знание о выходных
 * и датах прилета обратно (в DTO такой инфы нет).
 */
export function convertPriceIndexDataToDynamicsData(
    dynamicsPrices: TAviaPriceIndexDynamicsData,
): TDynamicsDaysInfo {
    if (!dynamicsPrices) {
        return {};
    }

    return Object.keys(dynamicsPrices).reduce<TDynamicsDaysInfo>(
        (info, key) => {
            info[key] = getDynamicsDay(dynamicsPrices[key]);

            return info;
        },
        {},
    );
}

/**
 * Трансформирует данные из ответа ручки priceIndex в TDynamicTableDaysInfo
 * @param dynamicsPrices - данные ответа ручки
 * @param forSearchForm - поисковая форма из данных запроса ручки
 */
export function convertPriceIndexDataToDynamicTableDaysInfo(
    dynamicsPrices: TAviaPriceIndexDynamicsData,
    forSearchForm: TValidSearchForm,
): TDynamicTableDaysInfo {
    const dayDelta = getDatesDelta(forSearchForm);

    if (dayDelta === null) {
        return {};
    }

    return Object.keys(dynamicsPrices).reduce<TDynamicTableDaysInfo>(
        (info, dateForward) => {
            if (!isDateRobot(dateForward)) {
                return info;
            }

            const dateBackWard = getDateRobotFromMoment(
                moment(dateForward).add(dayDelta, 'days'),
            );
            const key = getKeyForTablePrice(dateForward, dateBackWard);

            info[key] = getDynamicsDay(dynamicsPrices[dateForward]);

            return info;
        },
        {},
    );
}

function getDynamicsDay(price: Nullable<IAviaPriceIndexPrice>): IDynamicsDay {
    if (!price) {
        return createFakePrice();
    }

    return {
        status: getStatus(price),
        progress: 100,
        price,
        fromSearch: false,
    };
}

/**
 * Мапит статусы из DTO (EAviaPriceIndexSearchDayPriceStatus) на статусы UI (EDynamicsDayStatus).
 */
function getStatus(day: IAviaPriceIndexPrice): EDynamicsDayStatus {
    if (day.roughly) {
        return EDynamicsDayStatus.SHOULD_SEARCH;
    }

    if (day.value) {
        return EDynamicsDayStatus.HAS_DATA;
    }

    return EDynamicsDayStatus.SHOULD_SEARCH;
}

export function createFakePrice(): IDynamicsDay {
    return {
        status: EDynamicsDayStatus.SHOULD_SEARCH,
        progress: 0,
        price: null,
        fromSearch: false,
    };
}
