import moment from 'moment';
import {batch} from 'react-redux';

import TValidDataForRequestDynamic from 'types/avia/dynamic/TValidDataForRequestDynamic';

import {CustomThunkAction} from 'reducers/trains/customDispatch';
import {inlineSearchRequest} from 'reducers/avia/aviaPriceIndex/dynamics/inlineSearchRequest';
import {convertPriceIndexDataToDynamicsData} from 'reducers/avia/aviaPriceIndex/utils/convertPriceIndexDataToDynamicsData';
import * as dynamicActions from 'reducers/avia/aviaPriceIndex/dynamics/actions';

import deviceTypeSelector from 'selectors/common/deviceTypeSelector';
import {aviaWeekPricesSearchFormSelector} from 'selectors/avia/weekPrices/aviaWeekPricesSearchFormSelector';
import {aviaWeekPricesFiltersSelector} from 'selectors/avia/weekPrices/aviaWeekPricesFiltersSelector';
import aviaWeekPricesCurrentRequestParamsSelector from 'selectors/avia/weekPrices/aviaWeekPricesCurrentRequestParamsSelector';
import aviaDynamicsPricesCurrentRequestParamsSelector from 'selectors/avia/dynamicsPrices/aviaDynamicsPricesCurrentRequestParamsSelector';
import {aviaWeekPricesPriceSelector} from 'selectors/avia/weekPrices/aviaWeekPricesSelectors';

import dynamicsWeekDates, {
    RANGE_DATES,
} from 'projects/avia/components/Dynamics/utilities/dynamicsWeekDates';
import getDateRobotFromMoment from 'utilities/dateUtils/getDateRobotFromMoment';
import getDatesDelta from 'projects/avia/lib/dynamic/getDatesDelta';
import isCompatibleDynamicSearch from 'projects/avia/lib/dynamic/isCompatibleDynamicSearch';
import getNearestIndexes from 'projects/avia/lib/dynamic/getNearestIndexes';
import isValidSearchForm from 'projects/avia/lib/dynamic/isValidSearchForm';
import {unknownErrToString} from 'utilities/error';

import {MOBILE_DYNAMIC_NEAREST_DATES_TO_SHOW} from 'projects/avia/pages/AviaSearchResult/components/AviaSubscriptionAndDynamicAsButtons/Days/Days';

import {aviaPriceIndexBrowserProvider} from 'serviceProvider/avia/aviaPriceIndexBrowserProvider';

import * as actions from './actions';

/**
 * Запрашивает данные динамики цен для блока с ценами на неделю на странице поиска
 */
export default function weekPricesRequest(
    dataForRequest: TValidDataForRequestDynamic,
): CustomThunkAction<void> {
    return async (dispatch, getState): Promise<void> => {
        const {searchForm, filters, interval} = dataForRequest;
        const state = getState();

        const {isMobile} = deviceTypeSelector(state);

        dispatch(actions.weekPricesSearchRequest(dataForRequest));

        try {
            const result = await aviaPriceIndexBrowserProvider.dynamics(
                interval,
                searchForm,
                filters,
            );

            const currentState = getState();
            const currentWeekPriceRequestParams =
                aviaWeekPricesCurrentRequestParamsSelector(currentState);
            const currentDynamicsPriceRequestParams =
                aviaDynamicsPricesCurrentRequestParamsSelector(currentState);
            const weekPricesCompatible =
                !currentWeekPriceRequestParams ||
                isCompatibleDynamicSearch(
                    dataForRequest,
                    currentWeekPriceRequestParams,
                );
            const dynamicPricesCompatible =
                !currentDynamicsPriceRequestParams ||
                isCompatibleDynamicSearch(
                    dataForRequest,
                    currentDynamicsPriceRequestParams,
                );

            const data = convertPriceIndexDataToDynamicsData(result);

            batch(() => {
                if (weekPricesCompatible) {
                    dispatch(
                        actions.weekPricesSearchSuccess({
                            data,
                            ...dataForRequest,
                        }),
                    );
                }

                if (dynamicPricesCompatible) {
                    dispatch(
                        dynamicActions.updatePrices({data, ...dataForRequest}),
                    );
                }
            });

            // Cтартуем поиски на ближайшие две даты к текущему дню,
            // если цены не известны или приблизительные.
            const actualState = getState();
            const prices = aviaWeekPricesPriceSelector(actualState);
            const actualSearchForm =
                aviaWeekPricesSearchFormSelector(actualState);

            if (!isValidSearchForm(actualSearchForm)) {
                return;
            }

            const actualFilters = aviaWeekPricesFiltersSelector(actualState);
            const dayDelta = getDatesDelta(actualSearchForm);
            const {when} = actualSearchForm;

            const dates = dynamicsWeekDates(when);
            const indexOfWhen = dates.findIndex(date => date === when);
            const nearestDayWithHeat = isMobile
                ? MOBILE_DYNAMIC_NEAREST_DATES_TO_SHOW
                : RANGE_DATES;
            const {start, end} = getNearestIndexes(
                dates,
                indexOfWhen,
                nearestDayWithHeat,
            );

            const datesForSearch = dates
                .slice(start, end + 1)
                .filter(date => date !== when)
                .filter(
                    date =>
                        !prices ||
                        !prices[date] ||
                        !prices[date].price ||
                        prices[date].price?.roughly,
                );

            batch(() => {
                datesForSearch.forEach(date => {
                    dispatch(
                        inlineSearchRequest(
                            date,
                            dayDelta === null
                                ? null
                                : getDateRobotFromMoment(
                                      moment(date).add(dayDelta, 'day'),
                                  ),
                            actualSearchForm,
                            actualFilters,
                        ),
                    );
                });
            });
        } catch (e) {
            const error = unknownErrToString(e);

            dispatch(
                actions.weekPricesSearchFailure({error, ...dataForRequest}),
            );
            console.log(e);
        }
    };
}
