import omit from 'lodash/omit';

import {Request, Response} from '@yandex-data-ui/core/lib/types';
import {IAviaGetFlightPageServiceParams} from 'server/services/AviaService/types/IAviaGetFlightPageServiceParams';
import {EHttpStatusCodes} from 'types/httpStatusCodes';

import {CustomThunkAction} from 'reducers/trains/customDispatch';
import {getFlightActions} from 'reducers/avia/flight/actions';

import {internalUrl} from 'utilities/url';
import {isUnknownAxiosError} from 'utilities/error';

import {aviaProvider} from 'serviceProvider/avia/aviaProvider';

export default function requestFlightThunkAction({
    req,
    res,
    iata,
    number,
    departureDate,
    from,
}: {
    req?: Request;
    res?: Response;
    iata: string;
    number: string;
    departureDate?: string;
    from?: string;
}): CustomThunkAction<void> {
    return async (dispatch, getState): Promise<void> => {
        const {
            avia: {flight: prevFlightInfo},
            common: {
                userInfo: {geoLocation},
            },
        } = getState();

        const {requestParams: prevRequestParams} = prevFlightInfo;

        const requestParams: IAviaGetFlightPageServiceParams = {
            iata,
            number,
            departureDate,
            from,
            userGeoId: geoLocation?.geoId,
        };

        if (
            prevRequestParams &&
            requestParams.iata === prevRequestParams.iata &&
            prevRequestParams.number === requestParams.number &&
            prevRequestParams.departureDate === requestParams.departureDate &&
            prevRequestParams.from === requestParams.from
        ) {
            return;
        }

        try {
            dispatch(getFlightActions.request(requestParams));

            const flightInfo = await aviaProvider
                .provider(req?.container)
                .getFlightPage(requestParams);

            dispatch(getFlightActions.success(flightInfo));
        } catch (err) {
            const status = isUnknownAxiosError(err)
                ? err.response?.status
                : undefined;

            dispatch(getFlightActions.failure(status));

            // TRAVELFRONT-4972: если из расписаний переходят с плохой датой, то
            // лучше показать рейс на ближайшие даты вместо 404
            if (
                req &&
                status === EHttpStatusCodes.NotFound &&
                'when' in req.query
            ) {
                res?.redirect(internalUrl(req.path, omit(req.query, 'when')));
            }
        }
    };
}
