import {isEmpty} from 'lodash';
import {createReducer} from 'typesafe-actions';

import {
    INormalizedPreparedTDAnswer,
    getInitialNormalizedPreparedTDAnswer,
} from 'reducers/avia/utils/ticketDaemon/normalizedPreparedTDAnswer';
import {TAviaOrderActions, aviaOrderActions} from 'reducers/avia/order/actions';
import {mergeTDAnswer} from 'reducers/avia/utils/ticketDaemon/mergeTDAnswer';
import {cleanUpOutdatedVariants} from 'reducers/avia/utils/ticketDaemon/cleanUpOutdatedVariants';

import {logWarning} from 'utilities/logger/logWarning';

export interface IAviaVariantOrder extends INormalizedPreparedTDAnswer {
    qid: string | null;
    error: boolean;
    withBaggage: boolean;
    variantKey: string | null;
    useOrderData: boolean;
    partnersDataWereFetched: boolean;
    badPartners: string[];
    hasGoodPrice: boolean;
    // Флаг означающий, что данные могут содержать неактуальные цены
    dirtyPrices: boolean;
}

const INITIAL_STATE: IAviaVariantOrder = {
    ...getInitialNormalizedPreparedTDAnswer(),
    error: false,
    qid: null,
    withBaggage: false,
    variantKey: null,
    useOrderData: false,
    partnersDataWereFetched: false,
    badPartners: [],
    hasGoodPrice: false,
    dirtyPrices: false,
};

export default createReducer<IAviaVariantOrder, TAviaOrderActions>(
    INITIAL_STATE,
)
    .handleAction(aviaOrderActions.setBadPartners, (state, {payload}) => ({
        ...state,
        badPartners: payload,
    }))
    .handleAction(
        aviaOrderActions.setPartnersDataWereFetched,
        (state, {payload}) => ({
            ...state,
            cont: null,
            partnersDataWereFetched: payload,
        }),
    )
    .handleAction(aviaOrderActions.setSearchQid, (state, {payload}) => ({
        ...state,
        qid: payload,
    }))
    .handleAction(aviaOrderActions.setBaggageFilter, (state, {payload}) =>
        state.withBaggage === payload
            ? state
            : {
                  ...state,
                  withBaggage: payload,
              },
    )
    .handleAction(aviaOrderActions.setVariantKey, (state, {payload}) => ({
        ...state,
        variantKey: payload,
    }))
    .handleAction(aviaOrderActions.setUseOrderData, (state, {payload}) => ({
        ...state,
        useOrderData: payload,
    }))
    .handleAction(aviaOrderActions.setHasGoodPrice, (state, {payload}) => ({
        ...state,
        hasGoodPrice: payload,
    }))
    .handleAction(aviaOrderActions.setVariantData, (state, {payload}) => {
        // Не фильтруем неактуальные цены, чтобы отразить пользователю хоть что-то
        // В процессе опроса партнёров неактуальные цены будут подчищены
        const preparedData = mergeTDAnswer(state, payload, false);
        const {variantGroups} = preparedData;
        const groupKeys = Object.keys(variantGroups);

        // такой ситуации в теории быть не должно, но если она случится - мы должны быть в курсе
        if (groupKeys.length > 1 && typeof window !== 'undefined') {
            logWarning({
                message:
                    '[YATRAVEL][AVIA] Неоднозначные данные на странице покупки',
                block: 'aviaOrderReducer',
            });
        }

        return {
            ...state,
            ...preparedData,
            dirtyPrices: true,
            variantKey: groupKeys[0],
        };
    })
    .handleAction(aviaOrderActions.updateProgress, (state, {payload}) =>
        payload &&
        (state.progress.all !== payload.all ||
            state.progress.current !== payload.current)
            ? {
                  ...state,
                  progress: payload,
              }
            : state,
    )
    .handleAction(aviaOrderActions.updateVariant, (state, {payload}) => {
        let currentState = state;

        // Очищаем текущие данные от устаревших вариантов
        if (currentState.dirtyPrices) {
            const variants = {...currentState.variants};
            const variantGroups = {...currentState.variantGroups};

            cleanUpOutdatedVariants(
                currentState.partners,
                variants,
                variantGroups,
            );

            currentState = {
                ...currentState,
                variants,
                variantGroups,
            };
        }

        return {
            ...mergeTDAnswer(currentState, payload),
            dirtyPrices: false,
            partnersDataWereFetched: payload.cont === null,
        };
    })
    .handleAction(aviaOrderActions.reset, () => INITIAL_STATE)
    .handleAction(aviaOrderActions.updateVariantFailure, state => ({
        ...state,
        error: isEmpty(state.variantGroups),
    }));
