import {createReducer, ActionType} from 'typesafe-actions';

import * as Flags from 'utilities/flags/flags';
import IPrice from 'utilities/currency/PriceInterface';
import {usePrevIfAreEqual} from 'utilities/actions/usePrevIfAreEqual';

import * as actions from './actions';

export interface IAviaSearchBaggageFilter {
    enabled: boolean;
}

export interface IAviaSearchPlusPointsFilter {
    enabled: boolean;
}

export type TAviaSearchCompanyFilter = {
    companiesIds: number[];
    combinationsAreEnabled: boolean;
};

/* eslint-disable no-bitwise */
export enum EAviaSearchTransferFilter {
    NO_TRANSFERS = 1 << 0,
    ONE_CHANGE = 1 << 1,
    EXCLUDE_NIGHTLY = 1 << 2,
    NO_AIRPORT_CHANGE = 1 << 3,
}
/* eslint-enable no-bitwise */

/* eslint-disable no-bitwise */
export enum EAviaSearchTimeFilter {
    NIGHT = 1 << 0,
    MORNING = 1 << 1,
    DAY = 1 << 2,
    EVENING = 1 << 3,
}
/* eslint-enable no-bitwise */

export interface IAviaSearchTimeFilter {
    departure: Flags.Flag<EAviaSearchTimeFilter>;
    arrival: Flags.Flag<EAviaSearchTimeFilter>;
}

/** Минимальная и максимальная длительность пересадки в милисекундах */
export type TAviaSearchTransferRange = [number, number];

export interface IAviaSearchTransferFilter {
    value: Flags.Flag<EAviaSearchTransferFilter>;
    range: TAviaSearchTransferRange | null;
}

export interface IAviaSearchSegmentsAirportsFilter {
    /** Список идентификаторов аэропортов прибытия */
    arrival: number[];

    /** Список идентификаторов аэропортов отправления */
    departure: number[];

    /** Список идентификаторов аэропортов пересадок */
    transfers: number[];
}

export interface IAviaSearchSegmentsDirectionAirportsFilter {
    arrival: number[];
    departure: number[];
}

export interface IAviaSearchSegmentsTransferAirportsFilter {
    transfers: number[];
}

export type TAviaSearchAirportsFilterKeys =
    keyof IAviaSearchSegmentsAirportsFilter;

export interface IPriceFilter {
    value: IPrice | null;
}
export type TAviaSearchPartnersFilter = string[];

export interface IAviaSearchResultsFilters {
    baggage: IAviaSearchBaggageFilter;
    transfer: IAviaSearchTransferFilter;
    time: IAviaSearchTimeFilter[];
    company: TAviaSearchCompanyFilter;
    airports: IAviaSearchSegmentsAirportsFilter[];
    price: IPriceFilter;
    partners: TAviaSearchPartnersFilter;
    plusPoints: IAviaSearchPlusPointsFilter;
}

export type TAviaFilterKey = keyof IAviaSearchResultsFilters;

export function getInitialState(): IAviaSearchResultsFilters {
    return {
        baggage: {enabled: false},
        transfer: {
            value: Flags.createFlags<EAviaSearchTransferFilter>(),
            range: null,
        },
        time: [
            {
                arrival: Flags.createFlags<EAviaSearchTimeFilter>(),
                departure: Flags.createFlags<EAviaSearchTimeFilter>(),
            },
            {
                arrival: Flags.createFlags<EAviaSearchTimeFilter>(),
                departure: Flags.createFlags<EAviaSearchTimeFilter>(),
            },
        ],
        company: {
            companiesIds: [],
            combinationsAreEnabled: false,
        },
        airports: [
            {
                departure: [],
                arrival: [],
                transfers: [],
            },
            {
                departure: [],
                arrival: [],
                transfers: [],
            },
        ],
        price: {
            value: null,
        },
        partners: [],
        plusPoints: {
            enabled: false,
        },
    };
}

export default createReducer<
    IAviaSearchResultsFilters,
    ActionType<typeof actions>
>(getInitialState())
    .handleAction(actions.setBaggageFilter, (state, {payload}) => ({
        ...state,
        baggage: {enabled: payload},
    }))
    .handleAction(actions.setTransferFilter, (state, {payload}) => ({
        ...state,
        transfer: {
            ...state.transfer,
            value: payload,
        },
    }))
    .handleAction(actions.setTransferRangeFilter, (state, {payload}) => ({
        ...state,
        transfer: {...state.transfer, range: payload},
    }))
    .handleAction(actions.setTimeFilter, (state, {payload}) => ({
        ...state,
        time:
            payload.length === 1
                ? [
                      ...payload,
                      {
                          arrival: Flags.createFlags<EAviaSearchTimeFilter>(),
                          departure: Flags.createFlags<EAviaSearchTimeFilter>(),
                      },
                  ]
                : payload,
    }))
    .handleAction(actions.setCompanyFilter, (state, {payload}) => ({
        ...state,
        company: payload,
    }))
    .handleAction(actions.setDirectionAirportsFilter, (state, {payload}) => ({
        ...state,
        airports: [
            {
                ...state.airports[0],
                arrival: payload[0].arrival,
                departure: payload[0].departure,
            },
            {
                ...state.airports[1],
                arrival: payload[1].arrival,
                departure: payload[1].departure,
            },
        ],
    }))
    .handleAction(actions.setTransferAirportsFilter, (state, {payload}) => ({
        ...state,
        airports: [
            {
                ...state.airports[0],
                transfers: payload[0].transfers,
            },
            {
                ...state.airports[1],
                transfers: payload[1].transfers,
            },
        ],
    }))
    .handleAction(actions.setPriceFilter, (state, {payload}) => ({
        ...state,
        price: {value: payload},
    }))
    .handleAction(actions.setPartnersFilter, (state, {payload}) => ({
        ...state,
        partners: payload,
    }))
    .handleAction(actions.setPlusPointsFilter, (state, {payload}) => ({
        ...state,
        plusPoints: payload,
    }))
    .handleAction(actions.setFilterValues, (state, {payload}) =>
        usePrevIfAreEqual(state, {
            ...getInitialState(),
            ...payload,
        }),
    )
    .handleAction(actions.resetDirectionAirportsFilter, state => ({
        ...state,
        airports: [
            {
                ...state.airports[0],
                arrival: [],
                departure: [],
            },
            {
                ...state.airports[1],
                arrival: [],
                departure: [],
            },
        ],
    }))
    .handleAction(actions.resetTransferAirportsFilter, state => ({
        ...state,
        airports: [
            {
                ...state.airports[0],
                transfers: [],
            },
            {
                ...state.airports[1],
                transfers: [],
            },
        ],
    }))
    .handleAction(actions.resetTransferFilter, state => ({
        ...state,
        transfer: getInitialState()['transfer'],
        airports: [
            {
                ...state.airports[0],
                transfers: [],
            },
            {
                ...state.airports[1],
                transfers: [],
            },
        ],
    }))
    .handleAction(actions.resetFilterValue, (state, {payload: filterType}) => ({
        ...state,
        [filterType]: getInitialState()[filterType],
    }))
    .handleAction(actions.resetFilterValues, getInitialState);
