import {createReducer, ActionType} from 'typesafe-actions';

import {EAviaInitSearchError} from 'types/avia/search/EAviaInitSearchError';

import {
    getInitialNormalizedPreparedTDAnswer,
    INormalizedPreparedTDAnswer,
} from 'reducers/avia/utils/ticketDaemon/normalizedPreparedTDAnswer';
import {
    allFilterActions,
    TAllFilterAction,
} from 'reducers/avia/search/results/filters/filterActions';

import {ESortDirection} from 'projects/avia/lib/comparators/TComparator';
import {mergeTDAnswer} from '../../utils/ticketDaemon/mergeTDAnswer';
import {
    TAviaSearchFormParamsKey,
    getAviaSearchParamsKey,
} from 'projects/avia/lib/search/aviaSearchParamsKey';

import {IPartner} from 'server/api/AviaPartnersApi/AviaPartnersApi';

import {EAviaResultsSortType} from './EAviaResultsSortType';
import * as actions from './actions';
import filterReducer, {
    getInitialState as getFiltersInitialState,
    IAviaSearchResultsFilters,
} from './filters/reducer';
import * as popularPartnerActions from './partners/actions';
import * as popularFlightsActions from './popularFlights/actions';
import popularPartnersReducer, {
    getInitialState as getPopularPartnersInitialState,
    IAviaPopularPartnersState,
} from './partners/reducer';
import popularFlightsReducer, {
    IAviaPopularFlightsState,
    getInitialState as getPopularFlightsInitialState,
} from './popularFlights/reducer';

export interface IAviaSearchResultsSorting {
    type: EAviaResultsSortType;
    direction: ESortDirection;
}

export interface IAviaSearchResultsState extends INormalizedPreparedTDAnswer {
    searchKey: TAviaSearchFormParamsKey | null;
    qid: string | null;
    error: Nullable<EAviaInitSearchError>;
    partnersInfo: Record<string, IPartner>;
    sorting: IAviaSearchResultsSorting;
    filters: IAviaSearchResultsFilters;
    popularPartners: IAviaPopularPartnersState;
    popularFlights: IAviaPopularFlightsState;
}

const getInitialState = (): IAviaSearchResultsState => ({
    ...getInitialNormalizedPreparedTDAnswer(),
    searchKey: null,
    qid: null,
    error: null,
    partnersInfo: {},
    sorting: {
        type: EAviaResultsSortType.INTEREST,
        direction: ESortDirection.ASCENDING,
    },
    filters: getFiltersInitialState(),
    popularPartners: getPopularPartnersInitialState(),
    popularFlights: getPopularFlightsInitialState(),
});

export default createReducer<
    IAviaSearchResultsState,
    | ActionType<typeof actions>
    | TAllFilterAction
    | ActionType<typeof popularPartnerActions>
    | ActionType<typeof popularFlightsActions>
>(getInitialState())
    .handleAction(actions.setPartnersInfo, (state, {payload}) => ({
        ...state,
        partnersInfo: payload,
    }))
    .handleAction(actions.initSearchSuccess, (state, {payload}) => ({
        ...state,
        qid: payload.id,
    }))
    .handleAction(actions.initSearchFail, (state, {payload}) => ({
        ...state,
        error: payload,
        searchIsCompleted: true,
        cont: null,
    }))
    .handleAction(actions.setResultsSorting, (state, {payload}) => ({
        ...state,
        sorting: payload,
    }))
    .handleAction(actions.resetSearch, state => ({
        ...getInitialState(),
        partnersInfo: state.partnersInfo,
        sorting: state.sorting,
        filters: state.filters,
    }))
    .handleAction(actions.resetSearchKey, state => ({
        ...state,
        searchKey: null,
    }))
    .handleAction(actions.updateAviaSearchProgress, (state, {payload}) =>
        payload &&
        (state.progress.all !== payload.all ||
            state.progress.current !== payload.current)
            ? {
                  ...state,
                  progress: payload,
              }
            : state,
    )
    .handleAction(actions.initSearch, (state, {payload}) => ({
        ...getInitialState(),
        searchKey: getAviaSearchParamsKey(payload.query),
        partnersInfo: state.partnersInfo,
        sorting: state.sorting,
        filters: state.filters,
    }))
    .handleAction(actions.setTDAnswer, (state, {payload}) =>
        mergeTDAnswer(state, payload),
    )
    .handleAction(actions.stopSearch, state => ({
        ...state,
        searchIsCompleted: true,
        cont: null,
    }))
    .handleAction(allFilterActions, (state, action) => ({
        ...state,
        filters: filterReducer(state.filters, action),
    }))
    .handleAction(
        [
            popularPartnerActions.partnersFetchActions.request,
            popularPartnerActions.partnersFetchActions.success,
            popularPartnerActions.partnersFetchActions.failure,
        ],
        (state, action) => ({
            ...state,
            popularPartners: popularPartnersReducer(
                state.popularPartners,
                action,
            ),
        }),
    )
    .handleAction(
        [
            popularFlightsActions.flightsFetchActions.request,
            popularFlightsActions.flightsFetchActions.success,
            popularFlightsActions.flightsFetchActions.failure,
        ],
        (state, action) => ({
            ...state,
            popularFlights: popularFlightsReducer(state.popularFlights, action),
        }),
    );
