import {Dispatch} from 'redux';

import {ESubscriptionCode} from 'types/subscription/ESubscriptionCode';
import {ESubscriptionSource} from 'types/subscription/ESubscriptionSource';
import {ESubscriptionVerticalName} from 'types/subscription/ESubscriptionVerticalName';

import {StoreInterface} from 'reducers/storeTypes';
import {
    resetSubscription,
    setConfirmationStatus,
    setSubscription,
} from 'reducers/avia/search/subscription/actions';
import {subscribeOnPromo} from 'reducers/common/subscription/promoCodeInfo/thunk';

import {aviaFiltersViewDataSelector} from 'selectors/avia/search/filters/viewData';

import aviaBrowserProvider from 'serviceProvider/avia/aviaBrowserProvider';

export function subscribe(email: string) {
    return async function (
        dispatch: Dispatch,
        getState: () => StoreInterface,
    ): Promise<void> {
        await subscribeOnPrice(dispatch, getState, email);
        await subscribeOnPromo({
            email,
            promoSubscriptionCode: ESubscriptionCode.travelNews,
            source: ESubscriptionSource.PRICE_ALERTS_POPUP,
            travelVerticalName: ESubscriptionVerticalName.Avia,
        });
    };
}

async function subscribeOnPrice(
    dispatch: Dispatch,
    getState: () => StoreInterface,
    email: string,
): Promise<void> {
    const state = getState();
    const {
        avia: {
            aviaSearch: {
                results: {qid},
            },
        },
    } = state;
    const {minPrice} = aviaFiltersViewDataSelector.price(state);

    if (qid && minPrice) {
        const {id, needsConfirmation} =
            await aviaBrowserProvider.subscribeByEmail(qid, email, minPrice);

        dispatch(setConfirmationStatus(needsConfirmation));
        dispatch(setSubscription(id));
    }
}

export function unsubscribe(subscription: string) {
    return async function (
        dispatch: Dispatch,
        getState: () => StoreInterface,
    ): Promise<void> {
        const {
            avia: {
                aviaSearch: {
                    results: {qid},
                },
            },
        } = getState();

        if (qid) {
            await aviaBrowserProvider.unsubscribeByDirection(qid, subscription);
            dispatch(resetSubscription());
        }
    };
}
