import first from 'lodash/first';
import isEqual from 'lodash/isEqual';
import {batchActions} from 'redux-batched-actions';

import {ESearchFormFieldName} from 'components/SearchForm/types';
import {isNotUndefined} from 'types/utilities';
import {IAviaSearchFormPointField} from 'projects/avia/components/SearchForm/types';
import {Request} from '@yandex-data-ui/core/lib/types';

import {CustomThunkAction} from 'reducers/trains/customDispatch';
import {
    getSearchSuggestsActions,
    IAviaGetSearchSuggestsActionRequestParams,
} from 'reducers/avia/searchSuggests/actions';
import {
    setAviaSearchFormFromFieldAction,
    setAviaSearchFormToFieldAction,
} from 'reducers/avia/searchForm/actions';

import {getUserGeoLocation} from 'selectors/common/userInfoSelector';
import experimentsSelector from 'selectors/common/experimentsSelector';

import requestSearchSuggests from 'projects/avia/utilities/api/requestSearchSuggests';
import requestGeoLookup from 'projects/avia/utilities/api/requestGeoLookup';

import {ESuggestSource} from 'components/SearchSuggest/SearchSuggest';

function requestAndSetSuggestsThunkAction({
    fieldType,
    requestParams,
    needToSetPointToSearchForm,
    req,
}: {
    fieldType: ESearchFormFieldName.FROM | ESearchFormFieldName.TO;
    requestParams: IAviaGetSearchSuggestsActionRequestParams;
    needToSetPointToSearchForm: boolean;
    req?: Request;
}): CustomThunkAction<Promise<void>> {
    return async (dispatch, getState): Promise<void> => {
        try {
            dispatch(
                getSearchSuggestsActions.request({
                    fieldType,
                    requestParams,
                }),
            );

            const state = getState();
            const {aviaSearchToAnywherePage} = experimentsSelector(state);

            const {items} = await requestSearchSuggests({
                ...requestParams,
                field: fieldType,
                showCountries: fieldType === ESearchFormFieldName.TO,
                showAnywhere:
                    fieldType === ESearchFormFieldName.TO &&
                    aviaSearchToAnywherePage,
                req,
            });

            const {requestParams: currentRequestParams} =
                state.avia.searchSuggests[fieldType];

            /**
             * Если параллельно запустили еще один поиск, то устарелые данные нас не интересуют.
             */
            if (!isEqual(requestParams, currentRequestParams)) {
                return;
            }

            const firstSuggest = first(items);

            const setFormFieldAction =
                fieldType === ESearchFormFieldName.FROM
                    ? setAviaSearchFormFromFieldAction
                    : setAviaSearchFormToFieldAction;

            dispatch(
                batchActions(
                    [
                        getSearchSuggestsActions.success({
                            fieldType,
                            items,
                        }),
                        firstSuggest && needToSetPointToSearchForm
                            ? setFormFieldAction({
                                  inputValue: firstSuggest.title,
                                  source: ESuggestSource.SUGGESTS,
                                  selectedValue: firstSuggest,
                              })
                            : undefined,
                    ].filter(isNotUndefined),
                ),
            );
        } catch (err) {
            dispatch(getSearchSuggestsActions.failure({fieldType}));
        }
    };
}

export default function requestSearchSuggestsThunkAction({
    fromField,
    toField,
    needToSetByGeoPointIfPossible,
    req,
}: {
    fromField: IAviaSearchFormPointField;
    toField: IAviaSearchFormPointField;
    needToSetByGeoPointIfPossible: boolean;
    req?: Request;
}): CustomThunkAction<void> {
    return async (dispatch, getState): Promise<void> => {
        const state = getState();

        const geoLocation = getUserGeoLocation(state);

        let {geoName} = geoLocation;
        const {geoId} = geoLocation;

        if (needToSetByGeoPointIfPossible && !fromField.selectedValue) {
            // Возвращает город с аэропортом
            const geoLookupResponse = await requestGeoLookup({
                geoId,
                req,
            });

            if (geoLookupResponse) {
                geoName = geoLookupResponse.searchCity.title;
            }
        }

        const {
            searchSuggests: {
                from: {
                    requestParams: prevFromSuggestsRequestParams,
                    value: prevFromSuggests,
                },
                to: {
                    requestParams: prevToSuggestsRequestParams,
                    value: prevToSuggests,
                },
            },
        } = state.avia;

        const fromSuggestsRequestParams = {
            query: (
                fromField.inputValue ||
                (needToSetByGeoPointIfPossible ? geoName : '') ||
                ''
            ).toLowerCase(),
            clientCity: geoId || undefined,
            otherPoint: toField.selectedValue
                ? toField.selectedValue.pointKey
                : '',
        };

        if (
            !isEqual(
                prevFromSuggestsRequestParams,
                fromSuggestsRequestParams,
            ) ||
            !prevFromSuggests
        ) {
            const requestFromSuggestsPromise = dispatch(
                requestAndSetSuggestsThunkAction({
                    fieldType: ESearchFormFieldName.FROM,
                    requestParams: fromSuggestsRequestParams,
                    needToSetPointToSearchForm: Boolean(
                        needToSetByGeoPointIfPossible &&
                            !fromField.selectedValue &&
                            geoName,
                    ),
                    req,
                }),
            );

            /**
             * Если саджест "Откуда" нужно выставить по геопозиции, то нужно дождаться, пока он загрузится, т.к.
             * от него зависят параметры для запроса саджеста "Куда", иначе можно загружать параллельно.
             */
            if (needToSetByGeoPointIfPossible) {
                await requestFromSuggestsPromise;
            }
        }

        const {from: updatedFromField} = getState().avia.searchForm;

        const toSuggestsRequestParams = {
            query: (toField.inputValue || '').toLowerCase(),
            clientCity: geoId || undefined,
            otherPoint: updatedFromField.selectedValue
                ? updatedFromField.selectedValue.pointKey
                : '',
        };

        if (
            !isEqual(prevToSuggestsRequestParams, toSuggestsRequestParams) ||
            !prevToSuggests
        ) {
            await dispatch(
                requestAndSetSuggestsThunkAction({
                    fieldType: ESearchFormFieldName.TO,
                    requestParams: toSuggestsRequestParams,
                    needToSetPointToSearchForm: false,
                    req,
                }),
            );
        }
    };
}
