import {ActionType, createReducer} from 'typesafe-actions';

import EAsyncStatus from 'types/common/EAsyncStatus';
import {IFromPoint} from 'types/avia/landing/IFromPoint';
import {IAnywhereSettlementInfo} from 'types/avia/landing/ISettlementInfo';
import {IInlineSearchInfo} from 'types/avia/landing/IInlineSearchInfo';
import {TSettlementKey} from 'types/PointKey';
import {IAviaGetSearchToAnywherePageApiParams} from 'server/api/AviaGatewayApi/types/IAviaGetSearchToAnywherePageApiParams';

import * as actions from './actions';

export interface IAviaSearchToAnywhereState {
    // Статус запроса за данными к бекэнду
    status: EAsyncStatus | null;
    // Ошибка при запросе
    error: string | null;
    // Данные для запроса страницы
    requestParams: IAviaGetSearchToAnywherePageApiParams | null;
    // Данные города или станции отправления
    fromPoint: IFromPoint | null;
    // Информация о городах прибытия
    settlements: IAnywhereSettlementInfo[];
    // Информация о ценах из inline-поисков
    inlineSearches: Record<TSettlementKey, IInlineSearchInfo>;
}

const initialState: IAviaSearchToAnywhereState = {
    status: null,
    error: null,
    requestParams: null,
    fromPoint: null,
    settlements: [],
    inlineSearches: {},
};

export default createReducer<
    IAviaSearchToAnywhereState,
    ActionType<typeof actions>
>(initialState)
    .handleAction(actions.reset, () => initialState)
    .handleAction(actions.request, (state, {payload: {requestParams}}) => {
        return {
            ...initialState,
            status: EAsyncStatus.LOADING,
            requestParams,
        };
    })
    .handleAction(
        actions.requestSuccess,
        (
            state,
            {
                payload: {
                    requestParams,
                    response: {fromPoint, settlements},
                },
            },
        ) => {
            return {
                ...state,
                status: EAsyncStatus.SUCCESS,
                requestParams,
                fromPoint,
                settlements,
            };
        },
    )
    .handleAction(actions.requestFailure, (state, {payload: {error}}) => {
        return {
            ...state,
            status: EAsyncStatus.ERROR,
            error,
        };
    })
    .handleAction(
        actions.inlineSearchRequest,
        (state, {payload: {toPoint}}) => {
            return {
                ...state,
                inlineSearches: {
                    ...state.inlineSearches,
                    [toPoint]: {
                        status: EAsyncStatus.LOADING,
                        price: null,
                        toPoint,
                    },
                },
            };
        },
    )
    .handleAction(
        actions.inlineSearchSuccess,
        (
            state,
            {
                payload: {
                    inlineSearchInfo: {toPoint, status, price},
                },
            },
        ) => {
            return {
                ...state,
                inlineSearches: {
                    ...state.inlineSearches,
                    [toPoint]: {
                        status,
                        price,
                        toPoint,
                    },
                },
            };
        },
    )
    .handleAction(
        actions.inlineSearchFailure,
        (state, {payload: {toPoint}}) => {
            return {
                ...state,
                inlineSearches: {
                    ...state.inlineSearches,
                    [toPoint]: {
                        status: EAsyncStatus.ERROR,
                        price: null,
                        toPoint,
                    },
                },
            };
        },
    );
