import {batchActions} from 'redux-batched-actions';

import {Request, Response} from '@yandex-data-ui/core/lib/types';
import {IAviaSearchToAnywherePageParams} from 'projects/avia/pages/AviaSearchToAnywhere/types/IAviaSearchToAnywherePageParams';

import {
    CustomDispatch,
    CustomThunkAction,
    TGetState,
} from 'reducers/trains/customDispatch';
import * as actions from 'reducers/avia/searchToAnywhere/actions';
import {getAviaSearchFormActions} from 'reducers/avia/utils/fillSearchForm';
import {IAviaContext} from 'reducers/avia/data-types';
import {setAviaContext} from 'reducers/avia/context/actions';

import {getOneWayParam} from 'projects/avia/lib/urls/getSearchUrl';
import {getSuggestByPointKey} from 'projects/avia/lib/getSuggestByPointKey';
import {isCompatibleRequestParams} from 'projects/avia/lib/searchToAnywhere/isCompatibleRequestParams';
import {unknownErrToString} from 'utilities/error';
import {getSuggestForAnywhere} from 'projects/avia/lib/getSuggestForAnywhere';

import {aviaProvider} from 'serviceProvider/avia/aviaProvider';
import {aviaGeoProvider} from 'serviceProvider/avia/aviaGeoProvider';

interface IRequestSearchToAnywhereThunkAction
    extends IAviaSearchToAnywherePageParams {
    req?: Request;
    res?: Response;
}

export function requestSearchToAnywhereThunkAction({
    req,
    res,
    ...requestParams
}: IRequestSearchToAnywhereThunkAction): CustomThunkAction<void> {
    return async (dispatch, getState): Promise<void> => {
        await Promise.all([
            getSearchToAnywherePage(req, dispatch, getState, requestParams),
            prefillSearchFormAndContext(req, dispatch, requestParams),
        ]);
    };
}

async function getSearchToAnywherePage(
    req: Request | undefined,
    dispatch: CustomDispatch,
    getState: TGetState,
    requestParams: IAviaSearchToAnywherePageParams,
): Promise<void> {
    dispatch(actions.request({requestParams}));

    try {
        const aviaService = aviaProvider.provider(req?.container);
        const toAnywhereInfo = await aviaService.getSearchToAnywherePage(
            requestParams,
        );

        const {
            avia: {
                searchToAnywhere: {requestParams: currentRequestParams},
            },
        } = getState();

        if (
            currentRequestParams &&
            !isCompatibleRequestParams(currentRequestParams, requestParams)
        ) {
            return;
        }

        dispatch(
            actions.requestSuccess({
                requestParams,
                response: toAnywhereInfo,
            }),
        );
    } catch (err) {
        dispatch(actions.requestFailure({error: unknownErrToString(err)}));

        throw err;
    }
}

async function prefillSearchFormAndContext(
    req: Request | undefined,
    dispatch: CustomDispatch,
    requestParams: IAviaSearchToAnywherePageParams,
): Promise<void> {
    const geoService = aviaGeoProvider.provider(req?.container);
    const {
        from,
        passengers: {adults, children, infants},
        klass,
        dateForward,
        dateBackward,
    } = requestParams;

    const toSuggest = getSuggestForAnywhere();
    const fromSuggest = await getSuggestByPointKey(geoService, from);

    const aviaContext: IAviaContext = {
        from: fromSuggest,
        fromId: from,
        fromName: fromSuggest?.title || '',
        to: toSuggest,
        toId: toSuggest?.pointKey,
        toName: toSuggest?.title || '',
        adult_seats: String(adults),
        children_seats: String(children),
        infant_seats: String(infants),
        klass,
        when: dateForward,
        return_date: dateBackward || '',
        oneway: getOneWayParam(dateBackward),
    };

    dispatch(
        batchActions([
            ...getAviaSearchFormActions(aviaContext),
            setAviaContext(aviaContext),
        ]),
    );
}
