import {batch} from 'react-redux';

import {TSettlementKey} from 'types/PointKey';
import {IAviaGetSearchToCountryPageApiParams} from 'server/api/AviaGatewayApi/types/IAviaGetSearchToCountryPageApiParams';
import {IAviaParams} from 'server/services/AviaSearchService/types/IAviaParams';
import EAsyncStatus from 'types/common/EAsyncStatus';
import {IInlineSearchInfo} from 'types/avia/landing/IInlineSearchInfo';

import {CustomThunkAction} from 'reducers/trains/customDispatch';
import * as actions from 'reducers/avia/searchToCountry/actions';
import {AviaMinPriceFinder} from 'reducers/avia/aviaPriceIndex/utils/AviaMinPriceFinder';

import {currenciesConverterSelector} from 'selectors/common/currenciesSelector';
import {searchToCountryRequestParamsSelector} from 'selectors/avia/searchToCountry/searchToCountrySelectors';

import IPrice from 'utilities/currency/PriceInterface';
import {getOneWayParam} from 'projects/avia/lib/urls/getSearchUrl';
import {isCompatibleRequestParams} from 'projects/avia/lib/searchToCountry/isCompatibleRequestParams';

type TInlineSearchRequestAction = (
    // Данные с которыми была запрошена сама страница поиска в страну
    requestPageParams: IAviaGetSearchToCountryPageApiParams,
    // pointKey города прибытия, для которого запускается inline-поиск
    toPoint: TSettlementKey,
) => CustomThunkAction<void>;

export const inlineSearchRequest: TInlineSearchRequestAction = (
    requestPageParams,
    toPoint,
) => {
    return async (dispatch, getState): Promise<void> => {
        const state = getState();
        const currenciesConverter = currenciesConverterSelector(state);

        // Проверяет, что данный инлайн-поиск соответствует текущему контексту страницы
        const isActual = (): boolean => {
            const actualRequestPageParams =
                searchToCountryRequestParamsSelector(getState());

            return Boolean(
                actualRequestPageParams &&
                    isCompatibleRequestParams(
                        requestPageParams,
                        actualRequestPageParams,
                    ),
            );
        };

        if (!isActual()) {
            return;
        }

        dispatch(actions.inlineSearchRequest({toPoint}));

        const onPriceFounded = (
            price: IPrice | null,
            progress: number,
            finder: AviaMinPriceFinder,
            searchInProgress: boolean,
        ): void => {
            if (!isActual()) {
                return finder.stop();
            }

            const inlineSearchInfo: IInlineSearchInfo = {
                toPoint,
                status: searchInProgress
                    ? EAsyncStatus.LOADING
                    : EAsyncStatus.SUCCESS,
                price,
            };

            dispatch(actions.inlineSearchSuccess({inlineSearchInfo}));
        };

        const {
            passengers: {adults, children, infants},
            from,
            klass,
            dateForward,
            dateBackward,
        } = requestPageParams;

        const searchFormForInlineSearch: IAviaParams = {
            adult_seats: String(adults),
            children_seats: String(children),
            infant_seats: String(infants),
            fromId: from,
            toId: toPoint,
            klass: klass,
            when: dateForward,
            return_date: dateBackward || '',
            oneway: getOneWayParam(dateBackward),
        };

        const finder = new AviaMinPriceFinder({
            searchForm: searchFormForInlineSearch,
            filters: {},
            onPriceFounded: (price, progress, finderObj): void =>
                onPriceFounded(price, progress, finderObj, true),
            currenciesConverter,
        });

        try {
            const {price, progress} = await finder.search();

            onPriceFounded(price, progress, finder, false);
        } catch (e) {
            batch(() => {
                dispatch(actions.inlineSearchFailure({toPoint}));
            });
        }
    };
};
