import {ActionType, createReducer} from 'typesafe-actions';

import EAsyncStatus from 'types/common/EAsyncStatus';
import {IAviaGetSearchToCountryPageApiParams} from 'server/api/AviaGatewayApi/types/IAviaGetSearchToCountryPageApiParams';
import {IFromPoint} from 'types/avia/landing/IFromPoint';
import {IToCountry} from 'types/avia/landing/toCountry/IToCountry';
import {ISettlementInfo} from 'types/avia/landing/ISettlementInfo';
import {IInlineSearchInfo} from 'types/avia/landing/IInlineSearchInfo';
import {TSettlementKey} from 'types/PointKey';

import * as actions from './actions';

export interface IAviaSearchToCountryFromPoint extends IFromPoint {
    countryTitle: string;
}

export interface IAviaSearchToCountryState {
    // Статус запроса за данными к бекэнду
    status: EAsyncStatus | null;
    // Ошибка при запросе
    error: string | null;
    // Данные для запроса страницы
    requestParams: IAviaGetSearchToCountryPageApiParams | null;
    // Данные города или станции отправления
    fromPoint: IAviaSearchToCountryFromPoint | null;
    // Данные страны прибытия
    toCountry: IToCountry | null;
    // Информация о городах страны прибытия
    settlements: ISettlementInfo[];
    // Информация о ценах из inline-поисков
    inlineSearches: Record<TSettlementKey, IInlineSearchInfo>;
}

const initialState: IAviaSearchToCountryState = {
    status: null,
    error: null,
    requestParams: null,
    fromPoint: null,
    toCountry: null,
    settlements: [],
    inlineSearches: {},
};

export default createReducer<
    IAviaSearchToCountryState,
    ActionType<typeof actions>
>(initialState)
    .handleAction(actions.reset, () => initialState)
    .handleAction(actions.request, (state, {payload: {requestParams}}) => {
        return {
            ...initialState,
            status: EAsyncStatus.LOADING,
            requestParams,
        };
    })
    .handleAction(
        actions.requestSuccess,
        (
            state,
            {
                payload: {
                    requestParams,
                    response: {fromPoint, toCountry, settlements},
                    fromPointCountryTitle,
                },
            },
        ) => {
            return {
                ...state,
                status: EAsyncStatus.SUCCESS,
                requestParams,
                fromPoint: {
                    ...fromPoint,
                    countryTitle: fromPointCountryTitle,
                },
                toCountry,
                settlements,
            };
        },
    )
    .handleAction(actions.requestFailure, (state, {payload: {error}}) => {
        return {
            ...state,
            status: EAsyncStatus.ERROR,
            error,
        };
    })
    .handleAction(
        actions.inlineSearchRequest,
        (state, {payload: {toPoint}}) => {
            return {
                ...state,
                inlineSearches: {
                    ...state.inlineSearches,
                    [toPoint]: {
                        status: EAsyncStatus.LOADING,
                        price: null,
                        toPoint,
                    },
                },
            };
        },
    )
    .handleAction(
        actions.inlineSearchSuccess,
        (
            state,
            {
                payload: {
                    inlineSearchInfo: {toPoint, status, price},
                },
            },
        ) => {
            return {
                ...state,
                inlineSearches: {
                    ...state.inlineSearches,
                    [toPoint]: {
                        status,
                        price,
                        toPoint,
                    },
                },
            };
        },
    )
    .handleAction(
        actions.inlineSearchFailure,
        (state, {payload: {toPoint}}) => {
            return {
                ...state,
                inlineSearches: {
                    ...state.inlineSearches,
                    [toPoint]: {
                        status: EAsyncStatus.ERROR,
                        price: null,
                        toPoint,
                    },
                },
            };
        },
    );
